/* 3D draw test for XFreeR6 (Not use accelator)
   Code by JaeHyuk Cho <mailto:minzkn@minzkn.com> */

#include <sys/types.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <X11/Xlib.h>
#include <X11/Xthreads.h>

#if 1 /* Auto detect */
# define __DEF_DISPLAY__ (char *)0
#else /* Local machine */
# define __DEF_DISPLAY__ {"localhost:0.0"}
#endif

#if 0 /* Fast call */
#define __FC__ __attribute__((stdcall))
#else /* C call */
#define __FC__
#endif

struct ts_HandleX
{ 
 Display   *Display;
 Window     RootWindow, MainWindow;
 GC         GC;
 XGCValues  GCValues;
 int        x, y, w, h;
 int       *Break;
};

struct ts_ObjectLine
{
 struct ts_ObjectLine *Next;
 int                   Color;
 int                   x1, y1, z1, x2, y2, z2;
 int                   rx1, ry1, rx2, ry2;
};

int main(int s_Argc, char *s_Argv[]);

static double (mzapi_sin)(int s_angle);
static double (mzapi_cos)(int s_angle);

static struct ts_ObjectLine * __FC__ AddObjectLine(struct ts_ObjectLine *s_ObjectLine, int s_Color, int s_x1, int s_y1, int s_z1, int s_x2, int s_y2, int s_z2);
static struct ts_ObjectLine * __FC__ FreeObjectLine(struct ts_ObjectLine *s_ObjectLine);
static void __FC__ RenderingObjectLine(struct ts_ObjectLine *s_ObjectLine, int s_Distance, int s_FrontAngle, int s_SideAngle, int s_TopAngle);
static void __FC__ DrawObjectLine(struct ts_HandleX *s_Handle, struct ts_ObjectLine *s_ObjectLine, int s_Switch, int s_x, int s_y);
static struct ts_ObjectLine * __FC__ ObjectPuts(struct ts_ObjectLine *s_ObjectLine, int s_Color, const char *s_String, int s_x, int s_y, int s_z, int s_ScaleX, int s_ScaleY);

static void __BreakSignal__(int s_Signal);
static void __FC__ Sleeper(int s_mSleep);
static int __FC__ My3D(struct ts_HandleX *s_Handle);
static void *My3D_Thread(void *s_Argument);

static int __g_Break__ = 0;

static int __g_mzapi_sin90_table__[] = { /* sin(0) ~ sin(90) - fixed 16 floating point table */
 /*  0 */     0,  1143,  2287,  3429,  4571,  5711,  6850,  7986,  9120, 10252,
 /* 10 */ 11380, 12504, 13625, 14742, 15854, 16961, 18064, 19160, 20251, 21336,
 /* 20 */ 22414, 23486, 24550, 25606, 26655, 27696, 28729, 29752, 30767, 31772,
 /* 30 */ 32767, 33753, 34728, 35693, 36647, 37589, 38521, 39440, 40347, 41243,
 /* 40 */ 42125, 42995, 43852, 44695, 45525, 46340, 47142, 47929, 48702, 49460,
 /* 50 */ 50203, 50931, 51643, 52339, 53019, 53683, 54331, 54963, 55577, 56175,
 /* 60 */ 56755, 57319, 57864, 58393, 58903, 59395, 59870, 60326, 60763, 61183,
 /* 70 */ 61583, 61965, 62328, 62672, 62997, 63302, 63589, 63856, 64103, 64331,
 /* 80 */ 64540, 64729, 64898, 65047, 65176, 65286, 65376, 65446, 65496, 65526,
 /* 90 */ 65536
};

int main(int s_Argc, char *s_Argv[])
{
 const char *c_DisplayArea = __DEF_DISPLAY__;
 struct ts_HandleX s_Handle;

 fprintf(stdout, "3D box text for XFreeR6.\n");
 fprintf(stdout, "Code by JaeHyuk Cho <mailto:minzkn@minzkn.com>\n");
 fprintf(stdout, "\n - Ctrl+C key to quit.\n");

 (void)signal(SIGINT, __BreakSignal__);
 (void)signal(SIGHUP, __BreakSignal__);
 (void)signal(SIGPIPE, __BreakSignal__);

 s_Handle.Break = (int *)(&__g_Break__);

 (void)XInitThreads();

 s_Handle.Display = XOpenDisplay(s_Argc > 1 ? s_Argv[1] : c_DisplayArea);
 if(s_Handle.Display)
 {
  s_Handle.RootWindow = XDefaultRootWindow(s_Handle.Display);
  if(s_Handle.RootWindow)
  {
   unsigned int s_w, s_h;
   s_w = XDisplayWidth(s_Handle.Display, 0);
   s_h = XDisplayHeight(s_Handle.Display, 0);
   (void)fprintf(stdout, "Display resolution is %u x %u\n", s_w, s_h);
   
   s_Handle.x = s_w >> 2, s_Handle.y = s_h >> 2; 
   s_Handle.w = s_w >> 1, s_Handle.h = s_h >> 1;
   s_Handle.MainWindow = XCreateSimpleWindow(s_Handle.Display, s_Handle.RootWindow, 
                                             s_Handle.x, s_Handle.y, s_Handle.w, s_Handle.h, 1, 
					     WhitePixel(s_Handle.Display, 0), 
					     BlackPixel(s_Handle.Display, 0));
   #if 1 /* Override window */
   do
   {
    XSetWindowAttributes s_XSWA = {
     override_redirect: True
    };
    (void)XChangeWindowAttributes(s_Handle.Display, s_Handle.MainWindow, CWOverrideRedirect, &s_XSWA);
   }while(0);
   #endif
   
   (void)XSelectInput(s_Handle.Display, s_Handle.MainWindow, 
                      ExposureMask | KeyPressMask | ButtonPressMask | EnterWindowMask | LeaveWindowMask);
   
   (void)XMapWindow(s_Handle.Display, s_Handle.MainWindow);
   (void)XFlush(s_Handle.Display);
 
   s_Handle.GC = XCreateGC(s_Handle.Display, s_Handle.MainWindow, 0L, &s_Handle.GCValues);
   if(s_Handle.GC)
   {
    XEvent s_Event;
    
    xthread_fork(My3D_Thread, (void *)(&s_Handle));

    do
    { /* Event roll */
     (void)XNextEvent(s_Handle.Display, &s_Event);
     switch(s_Event.type)
     {
      case Expose:
           do
           {
            int s_Check, s_x, s_y;
	    unsigned int s_w, s_h;
            unsigned int s_depth, s_border;
            Window s_GetRootWindow;
            s_Check = XGetGeometry(s_Handle.Display, 
                                   s_Handle.MainWindow,
	 	                   &s_GetRootWindow,
		                   &s_x, &s_y, &s_w, &s_h,
	 	                   &s_border, &s_depth);
            if(s_Check != 0)
            {
             s_Handle.x = s_x, s_Handle.y = s_y; 
             s_Handle.w = s_w, s_Handle.h = s_h;
            }
           }while(0);
           break;
      case KeyPress:
	   switch(s_Event.xkey.keycode)
	   {
	    case 0x18: /* q */
	         *(s_Handle.Break) = 1;
	         break;
            default: 
                 fprintf(stdout, "Key press ! (%04xH)\n", s_Event.xkey.keycode);
	         break;
	   }
	   break;
      case ButtonPress:
           *(s_Handle.Break) = 1;
	   break;
      case EnterNotify:
           /* fprintf(stdout, "EnterNotify !\n"); */
           break;
      case LeaveNotify:
           /* fprintf(stdout, "Leave !\n"); */
           break;
      default: 
           fprintf(stdout, "Unknown event ! (%d)\n", s_Event.type);
           break;
     }
    }while(*(s_Handle.Break) == 0);

    while(*(s_Handle.Break) != (-1))Sleeper(10 /* ms */);
    (void)XFreeGC(s_Handle.Display, s_Handle.GC);
   }
   else fprintf(stdout, "Error: Can not open GC !\n");
   
   (void)XDestroyWindow(s_Handle.Display, s_Handle.MainWindow);
   (void)XFlush(s_Handle.Display);
  }
  else fprintf(stdout, "Error: Can not open root window !\n");
  (void)XCloseDisplay(s_Handle.Display);
 }
 else fprintf(stdout, "Error: Can not open display %s !\n", c_DisplayArea ? c_DisplayArea : "NULL");

 return(0);
}

#define __mzapi_sin90__(m_angle) __g_mzapi_sin90_table__[m_angle]

#define __mzapi_angle_degree_360__(m_angle) \
do \
{ \
 while((m_angle) < ((int)0))(m_angle) += (int)360; \
 while((m_angle) >= ((int)360))(m_angle) -= (int)360; \
}while(0)

#define __mzapi_fixed16_floating_point_to_double__(m_value) (((double)(m_value)) / ((double)65536.0))

static int (mzapi_sin_16fp)(int s_angle)
{
 __mzapi_angle_degree_360__(s_angle);
 if(s_angle <= 90)return(__mzapi_sin90__(s_angle));
 if(s_angle <= 180)return(__mzapi_sin90__(((int)180) - s_angle));
 if(s_angle <= 270)return(-__mzapi_sin90__(s_angle - ((int)180)));
 return(-__mzapi_sin90__(((int)360) - s_angle));
}

static int (mzapi_cos_16fp)(int s_angle)
{
 __mzapi_angle_degree_360__(s_angle);
 if(s_angle <= 90)return(__mzapi_sin90__(((int)90) - s_angle));
 if(s_angle <= 180)return(-__mzapi_sin90__(s_angle - ((int)90)));
 if(s_angle <= 270)return(-__mzapi_sin90__(((int)270) - s_angle));
 return(__mzapi_sin90__(s_angle - ((int)270)));
}

static double (mzapi_sin)(int s_angle)
{
 return(__mzapi_fixed16_floating_point_to_double__(mzapi_sin_16fp(s_angle)));
}

static double (mzapi_cos)(int s_angle)
{
 return(__mzapi_fixed16_floating_point_to_double__(mzapi_cos_16fp(s_angle)));
}

static struct ts_ObjectLine * __FC__ AddObjectLine(struct ts_ObjectLine *s_ObjectLine, int s_Color, int s_x1, int s_y1, int s_z1, int s_x2, int s_y2, int s_z2)
{
 struct ts_ObjectLine *s_New;
 s_New = (struct ts_ObjectLine *)malloc(sizeof(struct ts_ObjectLine));
 if(s_New)
 {
  s_New->Color = s_Color;
  s_New->x1 = s_x1, s_New->y1 = s_y1, s_New->z1 = s_z1;
  s_New->x2 = s_x2, s_New->y2 = s_y2, s_New->z2 = s_z2;
  
  s_New->rx1 = s_New->ry1 = s_New->rx2 = s_New->ry2 = 0;
  
  s_New->Next = s_ObjectLine;
  s_ObjectLine = s_New;
 }
 return(s_ObjectLine);
}

static struct ts_ObjectLine * __FC__ FreeObjectLine(struct ts_ObjectLine *s_ObjectLine)
{
 struct ts_ObjectLine *s_Trace;
 while(s_ObjectLine)
 { 
  s_Trace = s_ObjectLine;
  s_ObjectLine = s_ObjectLine->Next;
  free((void *)s_Trace);
 }
 return(s_ObjectLine);
}

static void __FC__ RotateObject(int *s_xx, int *s_yy, int *s_zz, int s_FrontAngle, int s_SideAngle, int s_TopAngle)
{
 int s_x, s_y, s_z, s_sx, s_sy, s_sz;

 s_x = (*(s_xx));
 s_y = (*(s_yy));
 s_z = (*(s_zz));
 if(s_FrontAngle > 0) 
 {
  s_sx =  ((s_x * mzapi_sin(s_FrontAngle)) - (s_y * mzapi_cos(s_FrontAngle)));
  s_sy = -((s_x * mzapi_cos(s_FrontAngle)) + (s_y * mzapi_sin(s_FrontAngle)));
  s_x = s_sx, s_y = s_sy;  
 }
 if(s_SideAngle > 0)
 {
  s_sy =  ((s_y * mzapi_sin(s_SideAngle)) - (s_z * mzapi_cos(s_SideAngle))); 
  s_sz = -((s_y * mzapi_cos(s_SideAngle)) + (s_z * mzapi_sin(s_SideAngle)));
  s_y = s_sy, s_z = s_sz; 
 }
 if(s_TopAngle > 0)
 {
  s_sz =  ((s_z * mzapi_sin(s_TopAngle)) - (s_x * mzapi_cos(s_TopAngle))); 
  s_sx = -((s_z * mzapi_cos(s_TopAngle)) + (s_x * mzapi_sin(s_TopAngle)));
  s_z = s_sz, s_x = s_sx;
 }
 *(s_xx) = s_x;
 *(s_yy) = s_y;
 *(s_zz) = s_z;
}

static void __FC__ RenderingObjectLine(struct ts_ObjectLine *s_ObjectLine, int s_Distance, int s_FrontAngle, int s_SideAngle, int s_TopAngle)
{
 int s_x, s_y, s_z;
 while(s_ObjectLine)
 {
  s_x = s_ObjectLine->x1;
  s_y = s_ObjectLine->y1;
  s_z = s_ObjectLine->z1;

  RotateObject(&s_x, &s_y, &s_z, s_FrontAngle, s_SideAngle, s_TopAngle);

  s_z -= s_Distance << 1;
  if(s_z == 0)s_z = 1;
  s_ObjectLine->rx1 = (s_Distance * s_x) / s_z;  
  s_ObjectLine->ry1 = (s_Distance * s_y) / s_z;  
 
  s_x = s_ObjectLine->x2;
  s_y = s_ObjectLine->y2;
  s_z = s_ObjectLine->z2;

  RotateObject(&s_x, &s_y, &s_z, s_FrontAngle, s_SideAngle, s_TopAngle);

  s_z -= s_Distance << 1;
  if(s_z == 0)s_z = 1;
  s_ObjectLine->rx2 = (s_Distance * s_x) / s_z;  
  s_ObjectLine->ry2 = (s_Distance * s_y) / s_z;  
 
  s_ObjectLine = s_ObjectLine->Next;
 }
}

static void __FC__ DrawObjectLine(struct ts_HandleX *s_Handle, struct ts_ObjectLine *s_ObjectLine, int s_Switch, int s_x, int s_y)
{
 if(s_Switch == 0)s_Handle->GCValues.foreground = BlackPixel(s_Handle->Display, 0);
 else s_Handle->GCValues.foreground = WhitePixel(s_Handle->Display, 0);
 (void)XChangeGC(s_Handle->Display, s_Handle->GC, GCForeground, &s_Handle->GCValues);
 while(s_ObjectLine)
 {
  (void)XDrawLine(s_Handle->Display, s_Handle->MainWindow, s_Handle->GC, 
                  s_x + s_ObjectLine->rx1, s_y + s_ObjectLine->ry1, 
                  s_x + s_ObjectLine->rx2, s_y + s_ObjectLine->ry2);
  s_ObjectLine = s_ObjectLine->Next;
 }
}

static struct ts_ObjectLine * __FC__ ObjectPuts(struct ts_ObjectLine *s_ObjectLine, int s_Color, const char *s_String, int s_x, int s_y, int s_z, int s_ScaleX, int s_ScaleY)
{
 static unsigned char __g_Font08__[] = {
  /* 0000 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 0010 */ 0x00, 0x00, 0x00, 0x00, 0x3E, 0x41, 0x55, 0x41, 0x41, 0x5D, 0x49, 0x41, 0x3E, 0x00, 0x00, 0x00,
  /* 0020 */ 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x6B, 0x7F, 0x7F, 0x77, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0030 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x7F, 0x7F, 0x7F, 0x7F, 0x3E, 0x1C, 0x08, 0x00, 0x00, 0x00,
  /* 0040 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x1C, 0x3E, 0x7F, 0x3E, 0x1C, 0x08, 0x00, 0x00, 0x00, 0x00,
  /* 0050 */ 0x00, 0x00, 0x00, 0x00, 0x08, 0x1C, 0x1C, 0x1C, 0x7F, 0x7F, 0x08, 0x08, 0x1C, 0x00, 0x00, 0x00,
  /* 0060 */ 0x00, 0x00, 0x00, 0x00, 0x08, 0x1C, 0x3E, 0x7F, 0x7F, 0x7F, 0x2A, 0x08, 0x08, 0x1C, 0x00, 0x00,
  /* 0070 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x3E, 0x3E, 0x1C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 0080 */ 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x77, 0x63, 0x63, 0x77, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F,
  /* 0090 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x22, 0x1C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 00a0 */ 0x7F, 0x7F, 0x7F, 0x77, 0x77, 0x63, 0x63, 0x41, 0x63, 0x63, 0x77, 0x77, 0x7F, 0x7F, 0x7F, 0x7F,
  /* 00b0 */ 0x00, 0x00, 0x00, 0x00, 0x3F, 0x07, 0x0F, 0x19, 0x3C, 0x66, 0x42, 0x66, 0x3C, 0x00, 0x00, 0x00,
  /* 00c0 */ 0x00, 0x00, 0x00, 0x3C, 0x66, 0x42, 0x66, 0x3C, 0x18, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x00, 0x00,
  /* 00d0 */ 0x00, 0x00, 0x00, 0x00, 0x30, 0x3C, 0x36, 0x33, 0x33, 0x36, 0x30, 0xF0, 0xE0, 0x00, 0x00, 0x00,
  /* 00e0 */ 0x00, 0x00, 0x00, 0x00, 0x3F, 0x33, 0x3F, 0x33, 0x33, 0x33, 0x37, 0x76, 0x60, 0x00, 0x00, 0x00,
  /* 00f0 */ 0x00, 0x00, 0x00, 0x08, 0x08, 0x6B, 0x3E, 0x36, 0x63, 0x36, 0x3E, 0x6B, 0x08, 0x08, 0x00, 0x00,
  /* 0100 */ 0x00, 0x00, 0x00, 0x00, 0x80, 0xC0, 0xE0, 0xF8, 0xFE, 0xF8, 0xE0, 0xC0, 0x80, 0x00, 0x00, 0x00,
  /* 0110 */ 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x07, 0x1F, 0x7F, 0x1F, 0x07, 0x03, 0x01, 0x00, 0x00, 0x00,
  /* 0120 */ 0x00, 0x00, 0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x00, 0x00, 0x00,
  /* 0130 */ 0x00, 0x00, 0x00, 0x00, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x33, 0x33, 0x00, 0x00, 0x00,
  /* 0140 */ 0x00, 0x00, 0x00, 0x00, 0x3F, 0x7B, 0x7B, 0x7B, 0x3B, 0x1B, 0x1B, 0x1B, 0x1B, 0x00, 0x00, 0x00,
  /* 0150 */ 0x00, 0x00, 0x00, 0x3E, 0x63, 0x30, 0x1C, 0x36, 0x63, 0x36, 0x1C, 0x06, 0x63, 0x3E, 0x00, 0x00,
  /* 0160 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x7F, 0x00, 0x00, 0x00,
  /* 0170 */ 0x00, 0x00, 0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x7E, 0x00, 0x00,
  /* 0180 */ 0x00, 0x00, 0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
  /* 0190 */ 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x00, 0x00, 0x00,
  /* 01a0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x18, 0x1C, 0xFE, 0x1C, 0x18, 0x10, 0x00, 0x00, 0x00, 0x00,
  /* 01b0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x18, 0x38, 0x7F, 0x38, 0x18, 0x08, 0x00, 0x00, 0x00, 0x00,
  /* 01c0 */ 0x00, 0x00, 0x00, 0x00, 0x42, 0x42, 0x42, 0x5A, 0xFF, 0x5A, 0x66, 0x42, 0x42, 0x00, 0x00, 0x00,
  /* 01d0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xFF, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 01e0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x3C, 0x3C, 0x7E, 0x7E, 0xFF, 0xDB,
  /* 01f0 */ 0xDB, 0xFF, 0x7E, 0x7E, 0x3C, 0x3C, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 0200 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 0210 */ 0x00, 0x00, 0x00, 0x18, 0x3C, 0x3C, 0x3C, 0x3C, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00, 0x00,
  /* 0220 */ 0x00, 0x00, 0x77, 0x77, 0x77, 0x66, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 0230 */ 0x00, 0x00, 0x06, 0x37, 0x3F, 0x7E, 0x76, 0x37, 0x3F, 0x7E, 0x76, 0x30, 0x00, 0x00, 0x00, 0x00,
  /* 0240 */ 0x00, 0x08, 0x08, 0x3E, 0x7F, 0x6B, 0x69, 0x7E, 0x3F, 0x4B, 0x6B, 0x7F, 0x3E, 0x08, 0x08, 0x00,
  /* 0250 */ 0x00, 0x00, 0x00, 0x71, 0x73, 0x77, 0x0E, 0x1C, 0x38, 0x77, 0x67, 0x47, 0x00, 0x00, 0x00, 0x00,
  /* 0260 */ 0x00, 0x00, 0x00, 0x38, 0x7C, 0x6C, 0x7C, 0x3B, 0x7F, 0x6E, 0x66, 0x7F, 0x3B, 0x00, 0x00, 0x00,
  /* 0270 */ 0x00, 0x00, 0x00, 0x1C, 0x1C, 0x1C, 0x38, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 0280 */ 0x00, 0x00, 0x00, 0x1C, 0x38, 0x70, 0x70, 0x70, 0x70, 0x70, 0x70, 0x38, 0x1C, 0x00, 0x00, 0x00,
  /* 0290 */ 0x00, 0x00, 0x00, 0x38, 0x1C, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x1C, 0x38, 0x00, 0x00, 0x00,
  /* 02a0 */ 0x00, 0x00, 0x00, 0x00, 0x18, 0xDB, 0xFF, 0x7E, 0x18, 0x7E, 0xFF, 0xDB, 0x18, 0x00, 0x00, 0x00,
  /* 02b0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7E, 0x7E, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  /* 02c0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x1C, 0x1C, 0x38, 0x00, 0x00,
  /* 02d0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 02e0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
  /* 02f0 */ 0x00, 0x00, 0x00, 0x01, 0x03, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0xE0, 0xC0, 0x80, 0x00, 0x00, 0x00,
  /* 0300 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x67, 0x6F, 0x7B, 0x73, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0310 */ 0x00, 0x00, 0x00, 0x1C, 0x3C, 0x3C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x3F, 0x3F, 0x00, 0x00, 0x00,
  /* 0320 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x67, 0x0E, 0x1C, 0x38, 0x70, 0x7F, 0x7F, 0x00, 0x00, 0x00,
  /* 0330 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x03, 0x1E, 0x1F, 0x03, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0340 */ 0x00, 0x00, 0x00, 0x0E, 0x1E, 0x3E, 0x76, 0x66, 0x66, 0x7F, 0x7F, 0x06, 0x06, 0x00, 0x00, 0x00,
  /* 0350 */ 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x60, 0x7E, 0x7F, 0x03, 0x03, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0360 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x60, 0x7E, 0x7F, 0x63, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0370 */ 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x07, 0x0E, 0x1C, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  /* 0380 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x63, 0x3E, 0x7F, 0x63, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0390 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x63, 0x7F, 0x3F, 0x03, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 03a0 */ 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x38, 0x00, 0x00, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00,
  /* 03b0 */ 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x38, 0x00, 0x00, 0x38, 0x38, 0x38, 0x70, 0x00, 0x00, 0x00,
  /* 03c0 */ 0x00, 0x00, 0x00, 0x00, 0x0E, 0x1C, 0x38, 0x70, 0xE0, 0x70, 0x38, 0x1C, 0x0E, 0x00, 0x00, 0x00,
  /* 03d0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x00, 0x7F, 0x7F, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 03e0 */ 0x00, 0x00, 0x00, 0x00, 0x70, 0x38, 0x1C, 0x0E, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0x00, 0x00, 0x00,
  /* 03f0 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x63, 0x07, 0x0E, 0x1C, 0x00, 0x1C, 0x1C, 0x00, 0x00, 0x00,
  /* 0400 */ 0x00, 0x00, 0x3E, 0x7F, 0x41, 0x5D, 0x5D, 0x55, 0x5F, 0x5F, 0x43, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0410 */ 0x00, 0x00, 0x00, 0x08, 0x1C, 0x3E, 0x77, 0x63, 0x63, 0x7F, 0x7F, 0x63, 0x63, 0x00, 0x00, 0x00,
  /* 0420 */ 0x00, 0x00, 0x00, 0x7E, 0x7F, 0x63, 0x63, 0x7E, 0x7F, 0x63, 0x63, 0x7F, 0x7E, 0x00, 0x00, 0x00,
  /* 0430 */ 0x00, 0x00, 0x00, 0x1E, 0x3F, 0x73, 0x60, 0x60, 0x60, 0x60, 0x73, 0x3F, 0x1E, 0x00, 0x00, 0x00,
  /* 0440 */ 0x00, 0x00, 0x00, 0x7C, 0x7E, 0x67, 0x63, 0x63, 0x63, 0x63, 0x67, 0x7E, 0x7C, 0x00, 0x00, 0x00,
  /* 0450 */ 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x60, 0x60, 0x7E, 0x7E, 0x60, 0x60, 0x7F, 0x7F, 0x00, 0x00, 0x00,
  /* 0460 */ 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x60, 0x60, 0x7E, 0x7E, 0x60, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
  /* 0470 */ 0x00, 0x00, 0x00, 0x1E, 0x3F, 0x73, 0x60, 0x60, 0x67, 0x67, 0x73, 0x3F, 0x1E, 0x00, 0x00, 0x00,
  /* 0480 */ 0x00, 0x00, 0x00, 0x63, 0x63, 0x63, 0x63, 0x7F, 0x7F, 0x63, 0x63, 0x63, 0x63, 0x00, 0x00, 0x00,
  /* 0490 */ 0x00, 0x00, 0x00, 0x3C, 0x3C, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x3C, 0x00, 0x00, 0x00,
  /* 04a0 */ 0x00, 0x00, 0x00, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x7E, 0x3C, 0x00, 0x00, 0x00,
  /* 04b0 */ 0x00, 0x00, 0x00, 0x63, 0x67, 0x6E, 0x7C, 0x78, 0x78, 0x7C, 0x6E, 0x67, 0x63, 0x00, 0x00, 0x00,
  /* 04c0 */ 0x00, 0x00, 0x00, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x7F, 0x7F, 0x00, 0x00, 0x00,
  /* 04d0 */ 0x00, 0x00, 0x00, 0x63, 0x77, 0x7F, 0x7F, 0x6B, 0x63, 0x63, 0x63, 0x63, 0x63, 0x00, 0x00, 0x00,
  /* 04e0 */ 0x00, 0x00, 0x00, 0x63, 0x63, 0x73, 0x7B, 0x7F, 0x6F, 0x67, 0x63, 0x63, 0x63, 0x00, 0x00, 0x00,
  /* 04f0 */ 0x00, 0x00, 0x00, 0x1C, 0x3E, 0x77, 0x63, 0x63, 0x63, 0x63, 0x77, 0x3E, 0x1C, 0x00, 0x00, 0x00,
  /* 0500 */ 0x00, 0x00, 0x00, 0x7E, 0x7F, 0x63, 0x63, 0x7F, 0x7E, 0x60, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
  /* 0510 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x63, 0x63, 0x63, 0x67, 0x7F, 0x3E, 0x07, 0x03, 0x00, 0x00,
  /* 0520 */ 0x00, 0x00, 0x00, 0x7E, 0x7F, 0x63, 0x63, 0x7F, 0x7E, 0x63, 0x63, 0x63, 0x63, 0x00, 0x00, 0x00,
  /* 0530 */ 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x60, 0x7E, 0x3F, 0x03, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0540 */ 0x00, 0x00, 0x00, 0x7E, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  /* 0550 */ 0x00, 0x00, 0x00, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0560 */ 0x00, 0x00, 0x00, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x77, 0x3E, 0x1C, 0x08, 0x00, 0x00, 0x00,
  /* 0570 */ 0x00, 0x00, 0x00, 0x63, 0x63, 0x63, 0x6B, 0x6B, 0x6B, 0x7F, 0x7F, 0x36, 0x36, 0x00, 0x00, 0x00,
  /* 0580 */ 0x00, 0x00, 0x00, 0x41, 0x63, 0x77, 0x3E, 0x1C, 0x1C, 0x3E, 0x77, 0x63, 0x41, 0x00, 0x00, 0x00,
  /* 0590 */ 0x00, 0x00, 0x00, 0x81, 0xC3, 0xE7, 0x7E, 0x3C, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  /* 05a0 */ 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0x60, 0x7F, 0x7F, 0x00, 0x00, 0x00,
  /* 05b0 */ 0x00, 0x00, 0x00, 0x3C, 0x3C, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x3C, 0x3C, 0x00, 0x00, 0x00,
  /* 05c0 */ 0x00, 0x00, 0x00, 0x80, 0xC0, 0xE0, 0x70, 0x38, 0x1C, 0x0E, 0x07, 0x03, 0x01, 0x00, 0x00, 0x00,
  /* 05d0 */ 0x00, 0x00, 0x00, 0x3C, 0x3C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x3C, 0x3C, 0x00, 0x00, 0x00,
  /* 05e0 */ 0x00, 0x08, 0x1C, 0x3E, 0x77, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 05f0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00,
  /* 0600 */ 0x00, 0x00, 0x78, 0x3C, 0x1E, 0x0F, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 0610 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x3E, 0x06, 0x3E, 0x7E, 0x66, 0x7E, 0x3E, 0x00, 0x00, 0x00,
  /* 0620 */ 0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x3E, 0x3F, 0x33, 0x33, 0x33, 0x3F, 0x3E, 0x00, 0x00, 0x00,
  /* 0630 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x60, 0x60, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0640 */ 0x00, 0x00, 0x00, 0x06, 0x06, 0x06, 0x3E, 0x7E, 0x66, 0x66, 0x66, 0x7E, 0x3E, 0x00, 0x00, 0x00,
  /* 0650 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x7F, 0x7F, 0x60, 0x7F, 0x3F, 0x00, 0x00, 0x00,
  /* 0660 */ 0x00, 0x00, 0x00, 0x1C, 0x3E, 0x36, 0x30, 0x7C, 0x7C, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
  /* 0670 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7E, 0x66, 0x66, 0x7E, 0x3E, 0x06, 0x66, 0x7E, 0x3C, 0x00,
  /* 0680 */ 0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x30, 0x3E, 0x3F, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
  /* 0690 */ 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  /* 06a0 */ 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x00, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x7E, 0x3C, 0x00,
  /* 06b0 */ 0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x33, 0x37, 0x3E, 0x3C, 0x3E, 0x37, 0x33, 0x00, 0x00, 0x00,
  /* 06c0 */ 0x00, 0x00, 0x00, 0x38, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  /* 06d0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0x7F, 0x7F, 0x6B, 0x6B, 0x6B, 0x63, 0x63, 0x00, 0x00, 0x00,
  /* 06e0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x3F, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
  /* 06f0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x63, 0x63, 0x63, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0700 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x3F, 0x33, 0x33, 0x3F, 0x3E, 0x30, 0x30, 0x30, 0x30, 0x00,
  /* 0710 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7E, 0x66, 0x66, 0x7E, 0x3E, 0x06, 0x06, 0x06, 0x06, 0x00,
  /* 0720 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x3F, 0x33, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
  /* 0730 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, 0x7F, 0x60, 0x7E, 0x3F, 0x03, 0x7F, 0x7E, 0x00, 0x00, 0x00,
  /* 0740 */ 0x00, 0x00, 0x00, 0x18, 0x18, 0x7E, 0x7E, 0x18, 0x18, 0x1B, 0x1B, 0x1F, 0x0E, 0x00, 0x00, 0x00,
  /* 0750 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x7F, 0x3E, 0x00, 0x00, 0x00,
  /* 0760 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x63, 0x63, 0x63, 0x77, 0x3E, 0x1C, 0x08, 0x00, 0x00, 0x00,
  /* 0770 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x63, 0x6B, 0x6B, 0x6B, 0x7F, 0x7F, 0x36, 0x00, 0x00, 0x00,
  /* 0780 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x63, 0x77, 0x3E, 0x1C, 0x3E, 0x77, 0x63, 0x41, 0x00, 0x00,
  /* 0790 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x63, 0x73, 0x3B, 0x1F, 0x0E, 0x1C, 0x38, 0x70, 0x00, 0x00,
  /* 07a0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x0E, 0x1C, 0x38, 0x70, 0x7F, 0x7F, 0x00, 0x00, 0x00,
  /* 07b0 */ 0x00, 0x00, 0x0E, 0x18, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0E, 0x00, 0x00, 0x00,
  /* 07c0 */ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  /* 07d0 */ 0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0E, 0x18, 0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00,
  /* 07e0 */ 0x00, 0x00, 0x00, 0x70, 0xFB, 0xDF, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  /* 07f0 */ 0x18, 0x18, 0x18, 0x3C, 0x24, 0x66, 0x42, 0xC3, 0x81, 0xC3, 0x42, 0x66, 0x24, 0x3C, 0x18, 0x18 
};

 int s_Byte, s_Index, s_Bit, s_CountX, s_ScaleCount, s_sx;
 unsigned char *s_CharBuffer;
 s_sx = s_x;
 do
 {
  s_Byte = *((unsigned char *)s_String);
  switch(s_Byte)
  {
   case '\n':
        s_x = s_sx;
        s_y += s_ScaleY << 4;
        break;
   default:
        s_CharBuffer = (unsigned char *)(&__g_Font08__[s_Byte << 4]);
        for(s_Index = 0;s_Index < 16;s_Index++)
        {
         for(s_Bit = 0, s_CountX = (8 - 1);s_Bit < 8;s_Bit++, s_CountX--)
         {
          if(s_CharBuffer[s_Index] & (1lu << s_Bit))
          {
           for(s_ScaleCount = 0;s_ScaleCount < s_ScaleY;s_ScaleCount++)
           {
            s_ObjectLine = AddObjectLine(s_ObjectLine, s_Color, 
             s_x + (s_CountX * s_ScaleX)                 , s_y + (s_Index * s_ScaleY) + s_ScaleCount, s_z, 
             s_x + (s_CountX * s_ScaleX) + (s_ScaleX - 1), s_y + (s_Index * s_ScaleY) + s_ScaleCount, s_z);
           }
          }
         }     
        } 
        s_x += s_ScaleX << 3;
        break;
  }
  s_String = ((char *)s_String) + 1;
 }while(s_Byte != '\0');
 return(s_ObjectLine);
}

static void __BreakSignal__(int s_Signal)
{
 __g_Break__ = 1; 
}

static void __FC__ Sleeper(int s_mSleep)
{
 struct timeval s_TimeVal; 
 s_TimeVal.tv_sec = s_mSleep / 1000;
 s_TimeVal.tv_usec = (s_mSleep % 1000) * 1000;
 select(0, (fd_set *)0, (fd_set *)0, (fd_set *)0, (struct timeval *)(&s_TimeVal));
}

static int __FC__ My3D(struct ts_HandleX *s_Handle)
{
 int s_White, s_Distance, s_Angle1, s_Angle2;
 struct ts_ObjectLine *s_Object = (struct ts_ObjectLine *)0;
 struct ts_ObjectLine *s_ObjectString = (struct ts_ObjectLine *)0;

 s_White = WhitePixel(s_Handle->Display, 0);
 s_Distance = 400;
 s_Angle1 = 0;
 s_Angle2 = 0;

 s_Object = AddObjectLine(s_Object, s_White, -100, -100, +100, +100, -100, +100);
 s_Object = AddObjectLine(s_Object, s_White, +100, -100, +100, +100, +100, +100);
 s_Object = AddObjectLine(s_Object, s_White, +100, +100, +100, -100, +100, +100);
 s_Object = AddObjectLine(s_Object, s_White, -100, +100, +100, -100, -100, +100);
 
 s_Object = AddObjectLine(s_Object, s_White, -100, -100, -100, +100, -100, -100);
 s_Object = AddObjectLine(s_Object, s_White, +100, -100, -100, +100, +100, -100);
 s_Object = AddObjectLine(s_Object, s_White, +100, +100, -100, -100, +100, -100);
 s_Object = AddObjectLine(s_Object, s_White, -100, +100, -100, -100, -100, -100);
 
 s_Object = AddObjectLine(s_Object, s_White, -100, -100, +100, -100, -100, -100);
 s_Object = AddObjectLine(s_Object, s_White, +100, -100, +100, +100, -100, -100);
 s_Object = AddObjectLine(s_Object, s_White, +100, +100, +100, +100, +100, -100);
 s_Object = AddObjectLine(s_Object, s_White, -100, +100, +100, -100, +100, -100);

 s_Object = AddObjectLine(s_Object, s_White, -200, -200, +200, +200, -200, +200);
 s_Object = AddObjectLine(s_Object, s_White, +200, -200, +200, +200, +200, +200);
 s_Object = AddObjectLine(s_Object, s_White, +200, +200, +200, -200, +200, +200);
 s_Object = AddObjectLine(s_Object, s_White, -200, +200, +200, -200, -200, +200);
  
 s_Object = AddObjectLine(s_Object, s_White, -200, -200, -200, +200, -200, -200);
 s_Object = AddObjectLine(s_Object, s_White, +200, -200, -200, +200, +200, -200);
 s_Object = AddObjectLine(s_Object, s_White, +200, +200, -200, -200, +200, -200);
 s_Object = AddObjectLine(s_Object, s_White, -200, +200, -200, -200, -200, -200);
 
 s_Object = AddObjectLine(s_Object, s_White, -200, -200, +200, -200, -200, -200);
 s_Object = AddObjectLine(s_Object, s_White, +200, -200, +200, +200, -200, -200);
 s_Object = AddObjectLine(s_Object, s_White, +200, +200, +200, +200, +200, -200);
 s_Object = AddObjectLine(s_Object, s_White, -200, +200, +200, -200, +200, -200);

 do
 { /* Bitmap */
  const char *c_Message[] = {
   "MINZKN's 3D box",
   "Test for Xorg",
   "minzkn@minzkn.com",
   (char *)0
  };
  int s_ScaleX, s_ScaleY, s_x, s_y, s_Line;

  s_ScaleX = 3;
  s_ScaleY = 8;
  
  s_Line = 0;
  while(c_Message[s_Line])s_Line++;
  
  s_y = (-8) * s_Line * s_ScaleY;
  s_Line = 0;
  while(c_Message[s_Line])
  {
   s_x = 0 - ((strlen(c_Message[s_Line]) * (s_ScaleX * 8)) >> 1);
   s_ObjectString = ObjectPuts(s_ObjectString, s_White, 
    c_Message[s_Line], s_x, s_y, -200, s_ScaleX, s_ScaleY);
   s_Line++;
   s_y += s_ScaleY << 4;
  }
 }while(0);

 /* For test KKOCKGGI */
 #if 0 
 s_Object = AddObjectLine(s_Object, s_White, -100, -100, +100, -110, -110, +100);
 #endif

 do
 {
  RenderingObjectLine(s_Object, s_Distance, s_Angle1, s_Angle1, s_Angle1);
  RenderingObjectLine(s_ObjectString, s_Distance, s_Angle2, s_Angle2, s_Angle2);
  
  DrawObjectLine(s_Handle, s_Object, 1, s_Handle->w >> 1, s_Handle->h >> 1);
  DrawObjectLine(s_Handle, s_ObjectString, 1, s_Handle->w >> 1, s_Handle->h >> 1);
  
  s_Angle1 = (s_Angle1 + 1) % 360;
  s_Angle2 = (s_Angle2 + 1) % 360;
  
  XFlush(s_Handle->Display); 
  
  Sleeper( 10 /* ms */ ); 
  #if 0 /* ü   ̰  ȿ */
  DrawObjectLine(s_Handle, s_Object, 0, s_Handle->w >> 1, s_Handle->h >> 1);
  DrawObjectLine(s_Handle, s_ObjectString, 0, s_Handle->w >> 1, s_Handle->h >> 1);
  #else /* ü  ü  ϴ° ȿ */
  XClearWindow(s_Handle->Display, s_Handle->MainWindow);
  #endif
 }while(*(s_Handle->Break) == 0);

 s_Object = FreeObjectLine(s_Object);
 s_ObjectString = FreeObjectLine(s_ObjectString);

 *(s_Handle->Break) = (-1);
 return(0);
}

static void *My3D_Thread(void *s_Argument)
{
 (void)My3D((struct ts_HandleX *)s_Argument);
 return(s_Argument);
}

/* End of source */
