/*
  Copyright (C) JAEHYUK CHO
  All rights reserved.
  Code by JaeHyuk Cho <mailto:minzkn@minzkn.com>
*/

#if !defined(__def_test_library_source_hash_c__)
#define __def_test_library_source_hash_c__ "hash.c"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sched.h>

#include "hash.h"

static t_mzapi_crc32 (mzapi_fastcall __mzhash_crc32__)(t_mzapi_crc32 s_crc32, unsigned char *s_buffer, int s_size);
static t_mzapi_hash_key (mzapi_fastcall __mzhash_function__)(struct ts_mzapi_hash *s_hash, const void *s_data, size_t s_size);
static int (mzapi_fastcall __mzhash_index__)(t_mzapi_hash_key s_key, int s_table_count);
static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_search_by_key__)(struct ts_mzapi_hash *s_hash, t_mzapi_hash_key s_key);
static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_next_search__)(struct ts_mzapi_hash *s_hash, struct ts_mzapi_hash_node *s_hash_node);
static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_prev_search__)(struct ts_mzapi_hash *s_hash, struct ts_mzapi_hash_node *s_hash_node);
static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_add__)(struct ts_mzapi_hash *s_hash, t_mzapi_hash_key s_key, t_mzapi_vector s_vector);
static int (mzapi_fastcall __mzhash_del__)(struct ts_mzapi_hash *s_hash, struct ts_mzapi_hash_node *s_hash_node);
static int (mzapi_fastcall __mzhash_info__)(struct ts_mzapi_hash *s_hash);

mzapi_export struct ts_mzapi_hash * (mzapi_fastcall mzapi_open_hash)(int s_table_count);
mzapi_export struct ts_mzapi_hash * (mzapi_fastcall mzapi_close_hash)(struct ts_mzapi_hash *s_hash);

static t_mzapi_crc32 (mzapi_fastcall __mzhash_crc32__)(t_mzapi_crc32 s_crc32, unsigned char *s_buffer, int s_size)
{
 static t_mzapi_crc32 sg_crc32_table[ 1 << 8 ];
 static int sg_given_crc32_table = 0;
 t_mzapi_crc32 s_value;
 int s_index, s_sub_index;
 if(sg_given_crc32_table == 0)
 { /* make crc32 table */
  sg_given_crc32_table = 1;
  for(s_index = 0;s_index < (int)((sizeof(sg_crc32_table) / sizeof(t_mzapi_crc32)));s_index++)
  {
   s_value = (t_mzapi_crc32)s_index;
   for(s_sub_index = 0;s_sub_index < 8;s_sub_index++)
   {
    if(s_value & 0x00000001ul)s_value = 0xedb88320ul ^ (s_value >> 1);
    else s_value >>= 1;
   }
   sg_crc32_table[s_index] = s_value; 
  }
 }
 s_value = s_crc32 ^ ((t_mzapi_crc32)0xfffffffful);    
 for(s_index = 0;s_index < s_size;s_index++)s_value = sg_crc32_table[(s_value ^ ((t_mzapi_crc32)s_buffer[s_index])) & ((t_mzapi_crc32)0x000000fful)] ^ (s_value >> 8);
 return(s_value ^ ((t_mzapi_crc32)0xfffffffful));
}

static t_mzapi_hash_key (mzapi_fastcall __mzhash_function__)(struct ts_mzapi_hash *s_hash, const void *s_data, size_t s_size)
{
 return((t_mzapi_hash_key)__mzhash_crc32__((s_hash != ((struct ts_mzapi_hash *)0)) ? s_hash->seed : ((t_mzapi_hash_key)0), (unsigned char *)s_data, s_size));
}

static int (mzapi_fastcall __mzhash_index__)(t_mzapi_hash_key s_key, int s_table_count)
{
 return((int)(s_key % ((t_mzapi_hash_key)s_table_count)));
}

static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_search_by_key__)(struct ts_mzapi_hash *s_hash, t_mzapi_hash_key s_key)
{
 struct ts_mzapi_hash_node *s_hash_node;
 if(s_hash == ((struct ts_mzapi_hash *)0))return((struct ts_mzapi_hash_node *)0);
 s_hash_node = s_hash->head[s_hash->index(s_key, s_hash->table_count)];
 while(s_hash_node != ((struct ts_mzapi_hash_node *)0))
 {
  if(s_hash_node->key == s_key)break;
  s_hash_node = s_hash_node->next;
 }
 return(s_hash_node);
}

static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_next_search__)(struct ts_mzapi_hash *s_hash, struct ts_mzapi_hash_node *s_hash_node)
{
 t_mzapi_hash_key s_key;
 if(s_hash_node != ((struct ts_mzapi_hash_node *)0))
 {
  s_key = s_hash_node->key;
  s_hash_node = s_hash_node->next;
  while(s_hash_node != ((struct ts_mzapi_hash_node *)0))
  {
   if(s_hash_node->key == s_key)break;
   s_hash_node = s_hash_node->next;
  }
 }
 return(s_hash_node);
}

static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_prev_search__)(struct ts_mzapi_hash *s_hash, struct ts_mzapi_hash_node *s_hash_node)
{
 t_mzapi_hash_key s_key;
 if(s_hash_node != ((struct ts_mzapi_hash_node *)0))
 {
  s_key = s_hash_node->key;
  s_hash_node = s_hash_node->prev;
  while(s_hash_node != ((struct ts_mzapi_hash_node *)0))
  {
   if(s_hash_node->key == s_key)break;
   s_hash_node = s_hash_node->prev;
  }
 }
 return(s_hash_node);
}

static struct ts_mzapi_hash_node * (mzapi_fastcall __mzhash_add__)(struct ts_mzapi_hash *s_hash, t_mzapi_hash_key s_key, t_mzapi_vector s_vector)
{
 struct ts_mzapi_hash_node *s_hash_node;
 int s_index;
 if(s_hash == ((struct ts_mzapi_hash *)0))return((struct ts_mzapi_hash_node *)0);
 s_hash_node = (struct ts_mzapi_hash_node *)malloc((size_t)sizeof(struct ts_mzapi_hash_node));
 if(s_hash_node != ((struct ts_mzapi_hash_node *)0))
 {
  s_hash_node->key = s_key;
  s_hash_node->vector = s_vector;
  s_index = s_hash->index(s_key, s_hash->table_count);
  if(s_hash->tail[s_index] == ((struct ts_mzapi_hash_node *)0))
  {
   s_hash_node->prev = s_hash_node->next = (struct ts_mzapi_hash_node *)0;
   s_hash->head[s_index] = s_hash_node;
   s_hash->tail[s_index] = s_hash_node;
  }
  else
  {
   s_hash_node->prev = s_hash->tail[s_index];
   s_hash_node->next = (struct ts_mzapi_hash_node *)0;
   s_hash->tail[s_index]->next = s_hash_node;
   s_hash->tail[s_index] = s_hash_node;
  }
 }
 return(s_hash_node);
}

static int (mzapi_fastcall __mzhash_del__)(struct ts_mzapi_hash *s_hash, struct ts_mzapi_hash_node *s_hash_node)
{
 int s_index;
 if(s_hash == ((struct ts_mzapi_hash *)0))return(-1);
 if(s_hash_node == ((struct ts_mzapi_hash_node *)0))return(0);
 s_index = s_hash->index(s_hash_node->key, s_hash->table_count);
 if(s_hash_node->prev == ((struct ts_mzapi_hash_node *)0))s_hash->head[s_index] = s_hash_node->next;
 else s_hash_node->prev->next = s_hash_node->next;
 if(s_hash_node->next == ((struct ts_mzapi_hash_node *)0))s_hash->tail[s_index] = s_hash_node->prev;
 else s_hash_node->next->prev = s_hash_node->prev;
 free((void *)s_hash_node);
 return(1);
}

static int (mzapi_fastcall __mzhash_info__)(struct ts_mzapi_hash *s_hash)
{
 int s_index;
 int s_hash_count;
 struct ts_mzapi_hash_node *s_hash_node;
 if(s_hash == ((struct ts_mzapi_hash *)0))return(-1);
 (void)fprintf(stdout,
  "\nhash info\n=========\n\n"
  "table_count = %d\n"
  "table_size  = %lu\n"
  "seed        = %08lXH\n\n",
  s_hash->table_count,
  (unsigned long)s_hash->table_size,
  (unsigned long)s_hash->seed);
 for(s_index = 0;s_index < s_hash->table_count;s_index++)
 {
  s_hash_count = 0;
  s_hash_node = s_hash->head[s_index];
  while(s_hash_node != ((struct ts_mzapi_hash_node *)0))
  {
   s_hash_count++;
   s_hash_node = s_hash_node->next;
  }
  (void)fprintf(stdout, " [%5d] hash_nodes=%d\n", s_index, s_hash_count);
 }
 (void)fputs("\nEnd of hash info.\n", stdout);
 return(1);
}

struct ts_mzapi_hash * (mzapi_fastcall mzapi_open_hash)(int s_table_count)
{
 struct ts_mzapi_hash *s_hash;
 int s_index;
 s_hash = (struct ts_mzapi_hash *)malloc((size_t)sizeof(struct ts_mzapi_hash));
 if(s_hash != ((struct ts_mzapi_hash *)0))
 {
  if(s_table_count <= 0)s_table_count = 256; /* default */
  s_hash->table_count   = s_table_count;
  s_hash->table_size    = (size_t)(s_hash->table_count * sizeof(struct ts_mzapi_hash_node *));
  s_hash->seed          = (t_mzapi_hash_key)0; /* default seed key */
  s_hash->head          = (struct ts_mzapi_hash_node **)malloc(s_hash->table_size);
  /* mapping member function */
  s_hash->function      = __mzhash_function__;
  s_hash->index         = __mzhash_index__;
  s_hash->search_by_key = __mzhash_search_by_key__;
  s_hash->next_search   = __mzhash_next_search__;
  s_hash->prev_search   = __mzhash_prev_search__;
  s_hash->add           = __mzhash_add__;
  s_hash->del           = __mzhash_del__;
  s_hash->info          = __mzhash_info__;
  /* initialize hash table */
  if(s_hash->head != ((struct ts_mzapi_hash_node **)0))
  {
   for(s_index = 0;s_index < s_hash->table_count;s_index++)s_hash->head[s_index] = (struct ts_mzapi_hash_node *)0;
   s_hash->tail = (struct ts_mzapi_hash_node **)malloc(s_hash->table_size);
   if(s_hash->tail != ((struct ts_mzapi_hash_node **)0))
   {
    for(s_index = 0;s_index < s_hash->table_count;s_index++)s_hash->tail[s_index] = (struct ts_mzapi_hash_node *)0;
    /* ok */
   }
   else s_hash = mzapi_close_hash(s_hash);
  }
  else
  {
   s_hash->tail = (struct ts_mzapi_hash_node **)0;
   s_hash = mzapi_close_hash(s_hash);
  }
 }
 return(s_hash);
}

struct ts_mzapi_hash * (mzapi_fastcall mzapi_close_hash)(struct ts_mzapi_hash *s_hash)
{
 int s_index;
 struct ts_mzapi_hash_node *s_prev_node;
 if(s_hash != ((struct ts_mzapi_hash *)0))
 {
  if(s_hash->tail != ((struct ts_mzapi_hash_node **)0))free((void *)s_hash->tail);
  if(s_hash->head != ((struct ts_mzapi_hash_node **)0))
  {
   for(s_index = 0;s_index < s_hash->table_count;s_index++)
   {
    while(s_hash->head[s_index] != ((struct ts_mzapi_hash_node *)0))
    {
     s_prev_node = s_hash->head[s_index];
     s_hash->head[s_index] = s_hash->head[s_index]->next;
     free((void *)s_prev_node);
    }
   }
   free((void *)s_hash->head);
  }
  free((void *)s_hash); 
 }
 return((struct ts_mzapi_hash *)0);
}

#endif

/* vim: set expandtab: */
/* End of source */
