/*
 Copyright (C) JAEHYUK CHO
 All rights reserved.
 
 Author: JaeHyuk Cho <minzkn@minzkn.com>
*/

/* ---- */

#if !defined(_ISOC99_SOURCE)
# define _ISOC99_SOURCE (1L)
#endif

#if !defined(_GNU_SOURCE)
# define _GNU_SOURCE (1L)
#endif

/* ---- */
# include <sys/types.h>
# include <sys/time.h>
# include <sys/socket.h>
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <unistd.h>

# include <netinet/in.h>
# include <netdb.h>
# include <arpa/inet.h>

/* ---- */

#if 1L
# define DEBUG (1)
#endif

/* ---- */

#if !defined(ethernet_header_t)
# pragma pack(push,1)
typedef struct ethernet_header_ts {
    uint8_t m_destination[6];
    uint8_t m_source[6];
    uint16_t m_type;
}__ethernet_header_t;
# pragma pack(pop)
# define ethernet_header_t __ethernet_header_t
# define ether_header_t ethernet_header_t
#endif


#if !defined(ip_header_t)
# pragma pack(push,1)
typedef struct ip_header_ts {
#  if __BYTE_ORDER == __LITTLE_ENDIAN
    uint8_t m_header_length : 4;
    uint8_t m_version : 4;
#  elif __BYTE_ORDER == __BIG_ENDIAN
    uint8_t m_version : 4;
    uint8_t m_header_length : 4;
#  else
    uint8_t m_version_with_header_length;
#  endif
    uint8_t m_type_of_service;
    uint16_t m_total_length;
    uint16_t m_identification;
    uint16_t m_fragment_offset;
  
    /* maybe with pseudo_tcp_header */
    uint8_t m_time_to_live;
    uint8_t m_protocol;
    uint16_t m_header_checksum;
    uint32_t m_source_address;
    uint32_t m_destination_address;
    /*
       optional field 
       ...
    */
}__ip_header_t;
# pragma pack(pop)
# define ip_header_t __ip_header_t
#endif

#if !defined(icmp_header_t)
# pragma pack(push,1)
typedef struct icmp_header_ts {
    uint8_t m_type;
    uint8_t m_code;
    uint16_t m_checksum;
}__icmp_header_t;
# pragma pack(pop)
# define icmp_header_t __icmp_header_t
#endif

#if !defined(icmp_redirect_message_t)
# pragma pack(push,1)
typedef struct icmp_redirect_message_ts {
    icmp_header_t m_header;
    uint32_t m_new_gateway_address;
    ip_header_t m_ip_header;
    uint8_t m_data[8];
}__icmp_redirect_message_t;
# pragma pack(pop)
# define icmp_redirect_message_t __icmp_redirect_message_t
#endif

/* ---- */

static unsigned int ip_checksum(const void *s_data, size_t s_size);
static int icmp_redirect_send(const char *s_gateway_address, const char *s_new_gateway_address, const char *s_sender_address, const char *s_destination_address);

int main(int s_argc, char **s_argv);

/* ---- */

#if defined(DEBUG)
static void * (__dump_packet_ex__)(size_t s_s, size_t s_d, void * s_da, size_t s_si);
static void * (__dump_packet_ex__)(size_t s_s, size_t s_d, void * s_da, size_t s_si)
{ size_t s_o, s_w, s_lo; unsigned char s_b[ 16 + 1 ];
 if(s_da == NULL)return(NULL); s_b[sizeof(s_b) - 1] = (unsigned char)'\0';
 for(s_o = (size_t)0;s_o < s_si;s_o += (size_t)16){
  s_w = ((s_si - s_o) <= ((size_t)16)) ? (s_si - s_o) : ((size_t)16);
  (void)fprintf(stderr, "%08lX", (unsigned long)(s_d + s_o));
  for(s_lo = (size_t)0;s_lo < s_w;s_lo++){
   if(s_lo == ((size_t)(16 >> 1)))(void)fputs(" | ", stderr);
   else (void)fputs(" ", stderr);
   s_b[s_lo] = *(((unsigned char *)s_da) + (s_s + s_o + s_lo));
   (void)fprintf(stderr, "%02X", (int)s_b[s_lo]);
   if((s_b[s_lo] & ((unsigned char)(1 << 7))) || (s_b[s_lo] < ((unsigned char)' ')) ||
      (s_b[s_lo] == ((unsigned char)0x7f)))s_b[s_lo] = (unsigned char)'.';}
  while(s_lo < ((size_t)16)){
   if(s_lo == ((size_t)(16 >> 1)))(void)fputs("     ", stderr);
   else (void)fputs("   ", stderr);
   s_b[s_lo] = (unsigned char)' '; s_lo++;}
  (void)fprintf(stderr, " [%s]\n", (char *)(&s_b[0]));}
 return(s_da);
}
static __inline void * (dump_packet_ex)(size_t s_seek_offset, size_t s_display_offset, void * s_data, size_t s_size);
static __inline void * (dump_packet_ex)(size_t s_seek_offset, size_t s_display_offset, void * s_data, size_t s_size)
{ return(__dump_packet_ex__(s_seek_offset, s_display_offset, s_data, s_size)); }
#define dump_packet(m_data,m_size) dump_packet_ex((size_t)0,(size_t)0,(void *)(m_data),(size_t)(m_size))
#endif

static unsigned int ip_checksum(const void *s_data, size_t s_size)
{
    register unsigned int s_result = 0u;

    while(s_size > ((size_t)1u)) {
        s_result += (unsigned int)(*((const uint16_t *)s_data));
        s_data = ((const uint16_t *)s_data) + ((size_t)1u);
        s_size -= (size_t)2u;
    }
   
    if(s_size > ((size_t)0u)) {
        s_result += (unsigned int)(*((const uint8_t *)s_data));
    }

    s_result = (s_result >> 16) + (s_result & 0xffffu);
    s_result += s_result >> 16;

    return((~s_result) & 0xffffu);
}

static int icmp_redirect_send(const char *s_gateway_address, const char *s_new_gateway_address, const char *s_sender_address, const char *s_destination_address)
{
    int s_result;

    struct in_addr s_gateway;
    struct in_addr s_new_gateway;
    struct in_addr s_sender;
    struct in_addr s_destination;

    size_t s_packet_size;
    uint8_t *s_packet;

    uint16_t s_identification;

    size_t s_ip_header_size;
    size_t s_icmp_redirect_message_size;

    ip_header_t s_ip_header;
    icmp_redirect_message_t s_icmp_redirect_message;

    int s_raw_socket;

    (void)inet_pton(AF_INET, s_gateway_address, (void *)(&s_gateway));
    (void)inet_pton(AF_INET, s_new_gateway_address, (void *)(&s_new_gateway));
    (void)inet_pton(AF_INET, s_sender_address, (void *)(&s_sender));
    (void)inet_pton(AF_INET, s_destination_address, (void *)(&s_destination));

    s_ip_header_size = ((sizeof(s_ip_header) + ((size_t)3u)) >> 2) << 2;
    s_icmp_redirect_message_size = sizeof(s_icmp_redirect_message);

    s_packet_size = s_ip_header_size + s_icmp_redirect_message_size;

    s_identification = htons((uint16_t)rand());

    (void)memset((void *)(&s_ip_header), 0, sizeof(s_ip_header));
#  if __BYTE_ORDER == __LITTLE_ENDIAN
    s_ip_header.m_header_length = (uint8_t)(s_ip_header_size >> 2);
    s_ip_header.m_version = (uint8_t)4u;
#  elif __BYTE_ORDER == __BIG_ENDIAN
    s_ip_header.m_version = (uint8_t)4u;
    s_ip_header.m_header_length = (uint8_t)(s_ip_header_size >> 2);
#  else
    s_ip_header.m_version_with_header_length = ((uint8_t)(s_ip_header_size >> 2)) | (((uint8_t)4u) << 4);
#  endif
    s_ip_header.m_type_of_service = (uint8_t)0u;
    s_ip_header.m_total_length = (uint16_t)htons((uint16_t)(s_packet_size));
    s_ip_header.m_identification = s_identification;
    s_ip_header.m_fragment_offset = (uint16_t)htons(0);
    s_ip_header.m_time_to_live = (uint8_t)64u /* IPDEFTTL */;
    s_ip_header.m_protocol = (uint8_t)1u /* IPPROTO_ICMP */;
    s_ip_header.m_source_address = (uint32_t)s_gateway.s_addr;
    s_ip_header.m_destination_address = (uint32_t)s_sender.s_addr;
    s_ip_header.m_header_checksum = (uint16_t)ip_checksum((void *)(&s_ip_header), sizeof(s_ip_header));

    (void)memset((void *)(&s_icmp_redirect_message), 0, sizeof(s_icmp_redirect_message));
    s_icmp_redirect_message.m_header.m_type = (uint8_t)5u;
    s_icmp_redirect_message.m_header.m_code = (uint8_t)1u;
    s_icmp_redirect_message.m_header.m_checksum = (uint16_t)0u;
    s_icmp_redirect_message.m_new_gateway_address = (uint32_t)s_new_gateway.s_addr;
#  if __BYTE_ORDER == __LITTLE_ENDIAN
    s_icmp_redirect_message.m_ip_header.m_header_length = (uint8_t)(s_ip_header_size >> 2);
    s_icmp_redirect_message.m_ip_header.m_version = (uint8_t)4u;
#  elif __BYTE_ORDER == __BIG_ENDIAN
    s_icmp_redirect_message.m_ip_header.m_version = (uint8_t)4u;
    s_icmp_redirect_message.m_ip_header.m_header_length = (uint8_t)(s_ip_header_size >> 2);
#  else
    s_icmp_redirect_message.m_ip_header.m_version_with_header_length = ((uint8_t)(s_ip_header_size >> 2)) | (((uint8_t)4u) << 4);
#  endif
    s_icmp_redirect_message.m_ip_header.m_type_of_service = (uint8_t)0u;
    s_icmp_redirect_message.m_ip_header.m_total_length = (uint16_t)htons((uint16_t)(s_packet_size));
    s_icmp_redirect_message.m_ip_header.m_identification = s_identification;
    s_icmp_redirect_message.m_ip_header.m_fragment_offset = (uint16_t)htons(0);
    s_icmp_redirect_message.m_ip_header.m_time_to_live = (uint8_t)64u /* IPDEFTTL */;
    s_icmp_redirect_message.m_ip_header.m_protocol = (uint8_t)0u;
    s_icmp_redirect_message.m_ip_header.m_source_address = (uint32_t)s_sender.s_addr;
    s_icmp_redirect_message.m_ip_header.m_destination_address = (uint32_t)s_destination.s_addr;
    s_icmp_redirect_message.m_ip_header.m_header_checksum = (uint16_t)ip_checksum((void *)(&s_icmp_redirect_message.m_ip_header), sizeof(s_icmp_redirect_message.m_ip_header));
    s_icmp_redirect_message.m_data[0] = (uint8_t)0u;
    s_icmp_redirect_message.m_data[1] = (uint8_t)0u;
    s_icmp_redirect_message.m_data[2] = (uint8_t)0u;
    s_icmp_redirect_message.m_data[3] = (uint8_t)0u;
    s_icmp_redirect_message.m_data[4] = (uint8_t)0u;
    s_icmp_redirect_message.m_data[5] = (uint8_t)0u;
    s_icmp_redirect_message.m_data[6] = (uint8_t)0u;
    s_icmp_redirect_message.m_data[7] = (uint8_t)0u;

    s_icmp_redirect_message.m_header.m_checksum = (uint16_t)ip_checksum((void *)(&s_icmp_redirect_message), sizeof(s_icmp_redirect_message));

    s_packet = (uint8_t *)malloc(s_packet_size);
    if(s_packet == ((void *)0)) {
        return(-1);
    }

    (void)memcpy(
        memset(
            (void *)(&s_packet[0]),
            0,
            s_ip_header_size
        ),
        (const void *)(&s_ip_header),
        sizeof(s_ip_header)
    );
    (void)memcpy(
        (void *)(&s_packet[s_ip_header_size]),
        (const void *)(&s_icmp_redirect_message),
        sizeof(s_icmp_redirect_message)
    );

#if defined(DEBUG)
    dump_packet(s_packet, s_packet_size);
#endif

    s_result = (-1);

    s_raw_socket = socket(PF_INET, SOCK_RAW, IPPROTO_RAW);
    if(s_raw_socket == (-1)) {
        perror("socket");
    }
    else {
        int s_value;
        struct sockaddr_in s_sockaddr_in;
        ssize_t s_send_bytes;
        
        s_value = 1;
        (void)setsockopt(s_raw_socket, IPPROTO_IP, IP_HDRINCL, (void *)(&s_value), (socklen_t)sizeof(s_value));

        (void)memset((void *)(&s_sockaddr_in), 0, sizeof(s_sockaddr_in));
        s_sockaddr_in.sin_family = AF_INET;
        s_sockaddr_in.sin_addr.s_addr = s_destination.s_addr;

        s_send_bytes = sendto(
            s_raw_socket,
            (const void *)s_packet,
            s_packet_size,
#if defined(MSG_NOSIGNAL)
            MSG_NOSIGNAL,
#else
            0,
#endif            
            (const struct sockaddr *)(&s_sockaddr_in),
            (socklen_t)sizeof(s_sockaddr_in)
        );
        if(s_send_bytes == ((ssize_t)(-1))) {
            perror("sendto");
        }
        else if(s_send_bytes == ((ssize_t)s_packet_size)) {
            s_result = 0;
        }

        (void)close(s_raw_socket);
    }

    free((void *)s_packet);

    return(s_result);
}

int main(int s_argc, char **s_argv)
{
    const char *s_gateway;
    const char *s_new_gateway;
    const char *s_sender;
    const char *s_destination;

    (void)fprintf(stdout, "icmp_redirect v0.0.1 - Code by JaeHyuk Cho <minzkn@minzkn.com>\n\n");

    if(s_argc < 5) {
        (void)fprintf(stdout,
            "usage: %s <gateway> <new_gateway> <sender> <destination>\n\n",
            s_argv[0]
        );
        
        return(EXIT_FAILURE);
    }
    
    s_gateway = s_argv[1];
    s_new_gateway = s_argv[2];
    s_sender = s_argv[3];
    s_destination = s_argv[4];

    (void)setuid(getuid());

    if(icmp_redirect_send(s_gateway, s_new_gateway, s_sender, s_destination) != 0) {
        (void)fprintf(stderr, "error!\n");
        return(EXIT_FAILURE);
    }

    (void)fprintf(stdout, "done.\n");

    return(EXIT_SUCCESS);
}

/* ---- */

/* vim: set expandtab: */
/* End of source */
