/*
 [ GPL ]

 Code by JaeHyuk Cho <minzkn@infoeq.com> 
*/

#if !defined(DEF_mp3_ext_c)
#define DEF_mp3_ext_c "mp3_ext.c"

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "mpg123.h"

#include "mp3_ext.h"

#if defined(DEF_I386)
#define DEF_USE_ASM (1)
#else
#define DEF_USE_ASM (0)
#endif

#define DEF_BUFFER_ALIGN ( 32 << 10 )

typedef enum
{
 E_PARSER_LEFT = (-1),
 E_PARSER_CENTER = 0,
 E_PARSER_RIGHT = 1,
}e_MZ_Parser;

int MZ_MP3_GetBitrate(unsigned long s_Version, unsigned long s_Layer, unsigned long s_BitrateIndex);
int MZ_MP3_GetFrequency(unsigned long s_Version, unsigned long s_FrequencyIndex);
char *MZ_MP3_GetGenre(unsigned char s_Index);

t_MP3 *MZ_CreateMP3(void);
t_MP3 *MZ_DestroyMP3(t_MP3 *s_MP3);
t_MP3 *MZ_PushMP3(t_MP3 *s_MP3, void *s_Buffer, int s_Size);
int MZ_DoLayerMP3(t_MP3_Frame *s_Frame, void **s_PCM, unsigned long *s_PCMSize);

unsigned long MZ_MP3_FileTime(char *s_MP3_Name);

void MZ_MP3_DecoderOpen(void);
void MZ_MP3_DecoderClose(void);
int MZ_MP3_DecoderPush(void *s_Buffer, int s_Length);
int MZ_MP3_DecoderPop(void *s_Buffer, int s_Length);
int MZ_MP3_DecoderFrequency(void);
int MZ_MP3_DecoderCheckBuffer(void);

void *MZ_Alloc(int s_Length);
void MZ_Free(void *s_Memory);
static t_MZ_Buffer *MZ_CreateBuffer(void);
t_MZ_Buffer *MZ_PushBuffer(t_MZ_Buffer *s_HANDLE_Buffer, void *s_Data, unsigned long s_Length);
unsigned long MZ_PopBuffer(t_MZ_Buffer *s_HANDLE_Buffer, void *s_Data, unsigned long s_Length);
static t_MZ_BufferNode *MZ_FlushBuffer(t_MZ_BufferNode *s_BufferNode);
t_MZ_Buffer *MZ_DestroyBuffer(t_MZ_Buffer *s_HANDLE_Buffer);
unsigned long MZ_CheckBuffer(t_MZ_Buffer *s_HANDLE_Buffer);

static void *MZ_AllocString(void *s_Data, int s_Length);
static int MZ_CompareString(void *s_Left, void *s_Right, int s_Length);
static void *MZ_SkipString(void *s_String, void *s_It, e_MZ_Parser s_Flag);

/* MP3    Bitrate  մϴ. */
int MZ_MP3_GetBitrate(unsigned long s_Version, unsigned long s_Layer, unsigned long s_BitrateIndex)
{
 int c_MP3TBL_Bitrate[4][4][16] = {
  { /* MPEG Version 2.5 */
   {  0,                                                                            }, /* Reserved */
   {  0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160, (-1) }, /* Layer III */
   {  0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160, (-1) }, /* Layer II */
   {  0,  32,  48,  56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256, (-1) }  /* Layer I */
  },
  { /* Reserved */
   {0,}, {0,}, {0,}, {0,}
  },
  { /* MPEG Version 2.0 (ISO / IEC 13818-3) */
   {  0,                                                                            }, /* Reserved */
   {  0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160, (-1) }, /* Layer III */
   {  0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160, (-1) }, /* Layer II */
   {  0,  32,  48,  56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256, (-1) }  /* Layer I */
  },
  { /* MPEG Version 1.0 (ISO / IEC 11172-3) */
   {  0,                                                                            }, /* Reserved */
   {  0,  32,  40,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, (-1) }, /* Layer III */
   {  0,  32,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384, (-1) }, /* Layer II */
   {  0,  32,  64,  96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, (-1) }  /* Layer I */
  } 
 };	
 if(s_Version >= 4 || s_Layer >= 4 || s_BitrateIndex >= 16)return(0);
 return(c_MP3TBL_Bitrate[(int)s_Version][(int)s_Layer][(int)s_BitrateIndex]);
}

/* MP3  ø ļ   ̺ Ͽ մϴ. ׷ ٸ 쵵 ִµ Դϴ. */
int MZ_MP3_GetFrequency(unsigned long s_Version, unsigned long s_FrequencyIndex)
{
 const int c_MP3TBL_Frequency[4][4] = {
     { 11025, 12000,  8000, (-1) }, /* MPEG Version 2.5 */
     { (-1), },                     /* Reserved */
     { 22050, 24000, 16000, (-1) }, /* MPEG Version 2.0 (ISO / IEC 13818-3) */
     { 44100, 48000, 32000, (-1) }  /* MPEG Version 1.0 (ISO / IEC 11172-3) */
    };
 if(s_Version >= 4 || s_FrequencyIndex >= 4)return(44100);
 return(c_MP3TBL_Frequency[(int)s_Version][(int)s_FrequencyIndex]);
}

/* ID3 header ǥõǴ 帣ȣ  ڿ ȯմϴ. */
char *MZ_MP3_GetGenre(unsigned char s_Index)
{
 char * const c_MP3TBL_Genre[] = {
  "Blues", "Classic Rock", "Country", "Dance", 
  "Disco", "Funk", "Grunge", "Hip-Hop", 
  "Jazz", "Metal", "New Age", "Oldies", 
  "Other", "Pop", "R&B", "Rap", 
  "Reggae", "Rock", "Techno", "Industrial", 
  "Alternative", "Ska", "Death Metal", "Pranks", 
  "Soundtrack", "Euro-Techno", "Ambient", "Trip-Hop",
  "Vocal", "Jazz+Funk", "Fusion", "Trance", 
  "Classical", "Instrumental", "Acid", "House", 
  "Game", "Sound Clip", "Gospel", "Noise", 
  "AlternRock", "Bass", "Soul", "Punk", 
  "Space", "Meditative", "Instrumental Pop", "Instrumental Rock", 
  "Ethnic", "Gothic", "Darkwave", "Techno-Industrial",
  "Electronic", "Pop-Folk", "Eurodance", "Dream", 
  "Southern Rock", "Comedy", "Cult", "Gangsta", 
  "Top 40", "Christian Rap", "Pop/Funk", "Jungle",
  "Native American", "Cabaret", "New Wave", "Psychadelic", 
  "Rave", "Showtunes", "Trailer", "Lo-Fi", 
  "Tribal", "Acid Punk", "Acid Jazz", "Polka",
  "Retro", "Musical", "Rock & Roll", "Hard Rock", 
  "Folk", "Folk/Rock", "National folk", "Swing", 
  "Fast-fusion", "Bebob", "Latin", "Revival", 
  "Celtic", "Bluegrass", "Avantgarde", "Gothic Rock", 
  "Progressive Rock", "Psychedelic Rock", "Symphonic Rock", "Slow Rock", 
  "Big Band", "Chorus", "Easy Listening", "Acoustic", 
  "Humour", "Speech", "Chanson", "Opera", 
  "Chamber Music", "Sonata", "Symphony", 
  "Booty Bass", "Primus", "Porn Groove", "Satire", 
  "Slow Jam", "Club", "Tango", "Samba", 
  "Folklore", "Ballad", "Powder Ballad", "Rhythmic Soul", 
  "Freestyle", "Duet", "Punk Rock", "Drum Solo", 
  "A Capella", "Euro-House", "Dance Hall", "Goa", 
  "Drum & Bass", "Club House", "Hardcore", "Terror", 
  "Indie", "BritPop", "NegerPunk", "Polsk Punk", 
  "Beat", "Christian Gangsta", "Heavy Metal", "Black Metal", 
  "Crossover", "Contemporary C", "Christian Rock", "Merengue", 
  "Salsa", "Thrash Metal", "Anime", "JPop", 
  "SynthPop"
 };
 if(s_Index >= (sizeof(c_MP3TBL_Genre) / sizeof(char *)))return("Unknown");
 return(c_MP3TBL_Genre[s_Index]);
}

/* Decoder ʿ ޸ Ҵ  ʱġ   */
t_MP3 *MZ_CreateMP3(void)
{
 t_MP3 *s_Return = (t_MP3 *)0;
 s_Return = MZ_Alloc(sizeof(t_MP3));
 if(s_Return)
 {
  s_Return->EncodeBuffer = (t_MP3_Frame *)0;
  s_Return->ChargeFrame  = (t_MP3_Frame *)0;
  s_Return->ChargeBuffer = (t_MZ_Buffer *)0;
  s_Return->Header       = 0lu;
  s_Return->DecodeLevel  = 0lu;
  s_Return->FrameCount   = 0lu;
  s_Return->PlayTime     = (unsigned long)0;
  s_Return->Title        = (char *)0;
  s_Return->Artist       = (char *)0;
  s_Return->Album        = (char *)0;
  s_Return->Year         = (char *)0;
  s_Return->Comment      = (char *)0;
  s_Return->Genre        = (char *)0;
  s_Return->IsID3        = 0;
 }
 return(s_Return);
}

/* Decoder 1/2 ۸ մϴ. */
static t_MP3_Frame *MZ_FreeMP3Frame(t_MP3_Frame *s_Frame)
{
 t_MP3_Frame *s_PrevFrame;
 while(s_Frame)
 {
  s_PrevFrame = s_Frame;
  s_Frame = s_Frame->Next;
  if(s_PrevFrame->Frame && s_PrevFrame->FrameSize > 0lu)MZ_Free(s_PrevFrame->Frame);
  if(s_PrevFrame->Decode && s_PrevFrame->DecodeSize > 0lu)MZ_Free(s_PrevFrame->Decode);
  MZ_Free(s_PrevFrame);
 } 
 return(s_Frame); 
}

/* Decoder Ҵ  ޸𸮸  մϴ. */
t_MP3 *MZ_DestroyMP3(t_MP3 *s_MP3)
{
 if(s_MP3)
 {
  if(s_MP3->EncodeBuffer)s_MP3->EncodeBuffer = MZ_FreeMP3Frame(s_MP3->EncodeBuffer);
  if(s_MP3->ChargeFrame )s_MP3->ChargeFrame  = MZ_FreeMP3Frame(s_MP3->ChargeFrame);
  if(s_MP3->ChargeBuffer)s_MP3->ChargeBuffer = MZ_DestroyBuffer(s_MP3->ChargeBuffer);	 
  if(s_MP3->Title       )MZ_Free(s_MP3->Title);
  if(s_MP3->Artist      )MZ_Free(s_MP3->Artist);
  if(s_MP3->Album       )MZ_Free(s_MP3->Album);
  if(s_MP3->Year        )MZ_Free(s_MP3->Year);
  if(s_MP3->Comment     )MZ_Free(s_MP3->Comment);
  if(s_MP3->Genre       )MZ_Free(s_MP3->Genre);
  MZ_Free(s_MP3);
  s_MP3 = (t_MP3 *)0;
 }	 
 return(s_MP3);
}

/* TAG unsigned long ϴ. */
#define DEF_MP3_TAG(a,b,c) (unsigned long)( \
		            ((unsigned long)(a) << 16) | \
		            ((unsigned long)(b) <<  8) | \
		            ((unsigned long)(c)      )   \
		           )

/* Decoder ۿ MP3 RAW ͸ ֽϴ. */
t_MP3 *MZ_PushMP3(t_MP3 *s_MP3, void *s_Buffer, int s_Size)
{
 if(s_MP3 == (t_MP3 *)0)s_MP3 = MZ_CreateMP3();
 if(s_MP3)
 {
  if(s_Buffer && s_Size > 0)
  {
   /* 1 FIFO ۿ ֽϴ. */
   s_MP3->ChargeBuffer = MZ_PushBuffer(s_MP3->ChargeBuffer, s_Buffer, s_Size);
L_DecodeProcess:;
   switch(s_MP3->DecodeLevel)
   {
    case 0lu: /* Search header :  ּ ϱ ... */
	 if(s_MP3->ChargeFrame)s_MP3->ChargeFrame = MZ_FreeMP3Frame(s_MP3->ChargeFrame);
	 if(MZ_CheckBuffer(s_MP3->ChargeBuffer) > 0)
	 {
          unsigned char s_Byte;
	  MZ_PopBuffer(s_MP3->ChargeBuffer, (void *)&s_Byte, sizeof(s_Byte));	  
          s_MP3->Header = (s_MP3->Header << 8) | (unsigned long)(s_Byte & 0xff); 
	  if((s_MP3->Header & 0x00fffffflu) == DEF_MP3_TAG('T', 'A', 'G'))s_MP3->DecodeLevel = 2lu;
	  else if((s_MP3->Header & DEF_MP3_FRAMESYNC) == DEF_MP3_FRAMESYNC)
	  {
           t_MP3_Frame *s_Frame;
           s_Frame = MZ_Alloc(sizeof(t_MP3_Frame));
           if(s_Frame)
	   {
            s_Frame->Next               = (t_MP3_Frame *)0;
            s_Frame->Header             = ((s_MP3->Header & 0xff      ) << 24) |
		                          ((s_MP3->Header & 0xff00    ) <<  8) |
					  ((s_MP3->Header & 0xff0000  ) >>  8) |
					  ((s_MP3->Header & 0xff000000) >> 24) ;		   
	    s_Frame->Version            = (s_MP3->Header >> 19) & 0x03;
	    s_Frame->Layer              = (s_MP3->Header >> 17) & 0x03;
	    s_Frame->Protection         = (s_MP3->Header >> 16) & 0x01;
	    s_Frame->BitrateIndex       = (s_MP3->Header >> 12) & 0x0f;
	    s_Frame->FrequencyIndex     = (s_MP3->Header >> 10) & 0x03;
	    s_Frame->Padding            = (s_MP3->Header >>  9) & 0x01;
	    s_Frame->FreeBit            = (s_MP3->Header >>  8) & 0x01;
	    s_Frame->ChannelMode        = (s_MP3->Header >>  7) & 0x03;
	    s_Frame->ModeExtension      = (s_MP3->Header >>  4) & 0x03;
	    s_Frame->Copyright          = (s_MP3->Header >>  3) & 0x01;
	    s_Frame->Original           = (s_MP3->Header >>  2) & 0x01;
	    s_Frame->Emphasis           = (s_MP3->Header      ) & 0x03;
            s_Frame->FrameSize          = 0lu;		    
	    s_Frame->Frame              = (void *)0;
	    s_Frame->DecodeSize         = 0lu;
	    s_Frame->Decode             = (void *)0;
            if(s_Frame->Version != 0x01        && /* Reserved version */
	       s_Frame->Layer != 0x00          && /* Reserved layer */
	       s_Frame->BitrateIndex != 0x00   && /* Free bitrate index */
	       s_Frame->BitrateIndex != 0x0f   && /* Bad bitrate index */
	       s_Frame->FrequencyIndex != 0x03
	      )
	    {
	     switch(s_Frame->Layer)
	     {
	      case 0x03: /* Layer I */
		   s_Frame->FrameSize  = 12000lu * MZ_MP3_GetBitrate(s_Frame->Version, s_Frame->Layer, s_Frame->BitrateIndex);
		   s_Frame->FrameSize /= MZ_MP3_GetFrequency(s_Frame->Version, s_Frame->FrequencyIndex);  
	    	   s_Frame->FrameSize  = ((s_Frame->FrameSize + s_Frame->Padding) << 2) - sizeof(t_MP3_Header);
		   break;	 
	      case 0x02: /* Layer II */
		   s_Frame->FrameSize  = 144000lu * MZ_MP3_GetBitrate(s_Frame->Version, s_Frame->Layer, s_Frame->BitrateIndex);
		   s_Frame->FrameSize /= MZ_MP3_GetFrequency(s_Frame->Version, s_Frame->FrequencyIndex);
		   s_Frame->FrameSize  = s_Frame->FrameSize + s_Frame->Padding - sizeof(t_MP3_Header);
		   break;
	      case 0x01: /* Layer III */
	      default:
		   s_Frame->FrameSize  = 144000lu * MZ_MP3_GetBitrate(s_Frame->Version, s_Frame->Layer, s_Frame->BitrateIndex);
		   s_Frame->FrameSize /= MZ_MP3_GetFrequency(s_Frame->Version, s_Frame->FrequencyIndex) << 
			                 (s_Frame->Version == 0x03 ? 0 : 1);
		   s_Frame->FrameSize  = s_Frame->FrameSize + s_Frame->Padding - sizeof(t_MP3_Header);
		   break;	   
	     }	    
	     if(s_Frame->FrameSize > 0lu)s_Frame->Frame = MZ_Alloc(s_Frame->FrameSize); 
             s_MP3->ChargeFrame = s_Frame;
	     s_MP3->Header = 0lu;
	     s_Frame = (t_MP3_Frame *)0; 
	     s_MP3->DecodeLevel = 1lu;
	    }
	    if(s_Frame)s_Frame = MZ_FreeMP3Frame(s_Frame);
	   }
	   else fprintf(stderr, "mp3_ext.c: MZ_PushMP3 - [ERROR] Can not allocate memory to s_Frame !!!\n");
	  }
	 }
         if(s_MP3->DecodeLevel != 0lu)goto L_DecodeProcess;	 
         if(MZ_CheckBuffer(s_MP3->ChargeBuffer) > 0)goto L_DecodeProcess;	 
	 break;
    case 1lu: /* Frame charge */
	 if(s_MP3->ChargeFrame)
	 {
	  if(s_MP3->ChargeBuffer)
	  {
           if(MZ_CheckBuffer(s_MP3->ChargeBuffer) >= s_MP3->ChargeFrame->FrameSize)
	   {
            if(s_MP3->ChargeFrame->FrameSize > 0lu)MZ_PopBuffer(s_MP3->ChargeBuffer, (void *)s_MP3->ChargeFrame->Frame, s_MP3->ChargeFrame->FrameSize);
	    s_MP3->ChargeFrame->FreeBit = 1; /* Check in frame */
	    if(s_MP3->EncodeBuffer)
	    {
             t_MP3_Frame *s_TraceFrame = s_MP3->EncodeBuffer;
	     while(s_TraceFrame->Next)s_TraceFrame = s_TraceFrame->Next;
	     s_TraceFrame->Next = s_MP3->ChargeFrame; 
	    }
	    else s_MP3->EncodeBuffer = s_MP3->ChargeFrame;
	    s_MP3->FrameCount++;
	    s_MP3->PlayTime  = s_MP3->FrameCount * 1152lu;
 	    s_MP3->PlayTime /= MZ_MP3_GetFrequency(s_MP3->ChargeFrame->Version, s_MP3->ChargeFrame->FrequencyIndex);
	    if(s_MP3->ChargeFrame->Version != 0x03)s_MP3->PlayTime >>= 1;
	    MZ_DoLayerMP3(s_MP3->ChargeFrame, &s_MP3->ChargeFrame->Decode, &s_MP3->ChargeFrame->DecodeSize); 
	    s_MP3->ChargeFrame = (t_MP3_Frame *)0; 
	    s_MP3->DecodeLevel = 0lu; 
            if(MZ_CheckBuffer(s_MP3->ChargeBuffer) > 0)goto L_DecodeProcess;	 
	   }
	  }
	 }
	 else s_MP3->DecodeLevel = 0lu; 
         if(s_MP3->DecodeLevel != 1lu)goto L_DecodeProcess;	 
         break;	
    case 0x02lu:
	 if(s_MP3->ChargeBuffer)
	 {
          if(MZ_CheckBuffer(s_MP3->ChargeBuffer) >= (128 - 3))
	  {
           unsigned char s_TAG[128] = {'T', 'A', 'G', 0, };
           if(s_MP3->Title  )MZ_Free(s_MP3->Title);
           if(s_MP3->Artist )MZ_Free(s_MP3->Artist);
           if(s_MP3->Album  )MZ_Free(s_MP3->Album);
           if(s_MP3->Year   )MZ_Free(s_MP3->Year);
           if(s_MP3->Comment)MZ_Free(s_MP3->Comment);
           if(s_MP3->Genre  )MZ_Free(s_MP3->Genre);
	   MZ_PopBuffer(s_MP3->ChargeBuffer, (void *)&s_TAG[3], sizeof(s_TAG) - 3); 
	   s_MP3->Title     = MZ_AllocString((void *)&s_TAG[ 3], 30); 
	   s_MP3->Artist    = MZ_AllocString((void *)&s_TAG[33], 30); 
	   s_MP3->Album     = MZ_AllocString((void *)&s_TAG[63], 30); 
	   s_MP3->Year      = MZ_AllocString((void *)&s_TAG[93],  4); 
	   s_MP3->Comment   = MZ_AllocString((void *)&s_TAG[97], 30); 
	   s_MP3->Genre     = MZ_AllocString((void *)MZ_MP3_GetGenre(s_TAG[127]), (-1));
           if(s_MP3->Title  )MZ_SkipString(s_MP3->Title  , " ", E_PARSER_RIGHT); 
           if(s_MP3->Artist )MZ_SkipString(s_MP3->Artist , " ", E_PARSER_RIGHT); 
           if(s_MP3->Album  )MZ_SkipString(s_MP3->Album  , " ", E_PARSER_RIGHT); 
           if(s_MP3->Year   )MZ_SkipString(s_MP3->Year   , " ", E_PARSER_RIGHT); 
           if(s_MP3->Comment)MZ_SkipString(s_MP3->Comment, " ", E_PARSER_RIGHT); 
           if(s_MP3->Genre  )MZ_SkipString(s_MP3->Genre  , " ", E_PARSER_RIGHT); 
	   s_MP3->DecodeLevel = 0lu;
           if(MZ_CheckBuffer(s_MP3->ChargeBuffer) > 0)goto L_DecodeProcess;	 
           s_MP3->IsID3 = 1;
	  }
	 } 
	 if(s_MP3->DecodeLevel != 2lu)goto L_DecodeProcess;	 
         break;	 
    default: /* Unknwon process : BUG */
	 /* fprintf(stderr, "mp3_ext.c: MZ_PushMP3 - [ERROR] Unknown decode level %lu !!!\n", s_MP3->DecodeLevel); */
	 s_MP3->DecodeLevel = 0lu;
         goto L_DecodeProcess;	 
   }
  }
  else 
  {
   if(s_MP3->ChargeFrame)s_MP3->ChargeFrame = MZ_FreeMP3Frame(s_MP3->ChargeFrame);	  
   if(s_MP3->ChargeBuffer)s_MP3->ChargeBuffer = MZ_DestroyBuffer(s_MP3->ChargeBuffer);	 
   s_MP3->Header = 0lu;
   s_MP3->DecodeLevel = 0lu;
   s_MP3->FrameCount = 0lu;
   s_MP3->PlayTime = (unsigned long)0;
  }
 }
 else fprintf(stderr, "mp3_ext.c: MZ_PushMP3 - [ERROR] s_MP3 is null !!!\n"); 
 return(s_MP3);
}

unsigned long MZ_MP3_FileTime(char *s_MP3_Name)
{
 unsigned long s_Return = 0lu;
 if(s_MP3_Name)
 {
  int s_Handle;
  s_Handle = open(s_MP3_Name, O_RDONLY);
  if(s_Handle >= 0)
  {
   int s_ReadSize;
   unsigned char s_Buffer[64 << 10];
   t_MP3 *s_MP3 = (t_MP3 *)0;
   do
   {
    s_ReadSize = read(s_Handle, &s_Buffer[0], sizeof(s_Buffer));
    if(s_ReadSize > 0)
    {
     s_MP3 = MZ_PushMP3(s_MP3, (void *)&s_Buffer[0], s_ReadSize);
    }
   }while(s_ReadSize > 0);
   close(s_Handle);	  
   if(s_MP3)
   {
    s_Return = (unsigned long)s_MP3->PlayTime;	   
    s_MP3 = MZ_DestroyMP3(s_MP3);
   }
  }
  else fprintf(stderr, "mp3_time.c: MZ_MP3_Time - [ERROR] Can not open mp3 \"%s\" !!!\n", s_MP3_Name);
 } 
 else fprintf(stderr, "mp3_time.c: MZ_MP3_Time - [ERROR] s_MP3_Name is null !!!\n");
 return(s_Return);
}

int MZ_DoLayerMP3(t_MP3_Frame *s_Frame, void **s_PCM, unsigned long *s_PCMSize)
{
 int s_Return = (-1);
 (void)s_Frame, (void)s_PCM, (void)s_PCMSize;
 if(s_PCM)*(s_PCM) = (void *)0;
 if(s_PCMSize)*(s_PCMSize) = 0lu;
 if(s_Frame)
 {
   
 }
 return(s_Return);
}

static struct mpstr g_MP3_DecoderHandle;
static t_MZ_Buffer *g_MP3_OutBuffer = (t_MZ_Buffer *)0;
static int g_MP3_Decoder_Frequency = (-1);

void MZ_MP3_DecoderOpen(void)
{
 if(g_MP3_OutBuffer)g_MP3_OutBuffer = MZ_DestroyBuffer(g_MP3_OutBuffer);
 g_MP3_Decoder_Frequency = (-1);
 InitMP3(&g_MP3_DecoderHandle);
}

void MZ_MP3_DecoderClose(void)
{
 ExitMP3(&g_MP3_DecoderHandle);	
 if(g_MP3_OutBuffer)g_MP3_OutBuffer = MZ_DestroyBuffer(g_MP3_OutBuffer);
}

int MZ_MP3_DecoderPush(void *s_Buffer, int s_Length)
{
 int s_Return = (-1);	
 int s_Check, s_FMBufferLength;
 unsigned char s_FMBuffer[32 << 10];
 if(s_Buffer && s_Length > 0)
 {
  s_Return = 0;	
  s_Check = decodeMP3(&g_MP3_DecoderHandle, s_Buffer, s_Length, (void *)&s_FMBuffer[0], &s_FMBufferLength); 
  while(s_Check == MP3_OK)
  {
   if(s_FMBufferLength > 0 && s_FMBufferLength <= sizeof(s_FMBuffer))
   {
    s_Return += s_FMBufferLength;
    g_MP3_OutBuffer = MZ_PushBuffer(g_MP3_OutBuffer, &s_FMBuffer[0], s_FMBufferLength);
    g_MP3_Decoder_Frequency = g_Frequency[g_MP3_DecoderHandle.fr.sampling_frequency];
   }
   s_Check = decodeMP3(&g_MP3_DecoderHandle, (void *)0, 0, (void *)&s_FMBuffer[0], &s_FMBufferLength);   
  }
 }
 else fprintf(stderr, "%s: %s - [ERROR] s_Buffer is null or s_Length(%d) <= 0 !!! ", __FILE__, __FUNCTION__, s_Length);
 return(s_Return); 
}

int MZ_MP3_DecoderPop(void *s_Buffer, int s_Length)
{
 int s_Return;
 if(g_MP3_OutBuffer)s_Return = MZ_PopBuffer(g_MP3_OutBuffer, s_Buffer, s_Length);
 else s_Return = 0;
 return(s_Return);
}

int MZ_MP3_DecoderFrequency(void)
{
 return(g_MP3_Decoder_Frequency);
}

int MZ_MP3_DecoderCheckBuffer(void)
{
 return(MZ_CheckBuffer(g_MP3_OutBuffer));
}

void *MZ_Alloc(int s_Length)
{
 if(s_Length > 0)return(malloc(s_Length));
 return((void *)0);
}

void MZ_Free(void *s_Memory)
{
 if(s_Memory)free(s_Memory);
 else fprintf(stderr, "%s: %s - [ERROR] s_Memory is null !!!\n", __FILE__, __FUNCTION__);
}

static t_MZ_Buffer *MZ_CreateBuffer(void)
{
 t_MZ_Buffer *s_Return;
 s_Return = (t_MZ_Buffer *)MZ_Alloc(sizeof(t_MZ_Buffer)); 
 if(s_Return)
 {
  s_Return->Total = 0lu; 
  s_Return->Align = (void *)0, s_Return->AlignIndex = 0; 
  s_Return->Head  = (t_MZ_BufferNode *)0; 
 }
 else fprintf(stderr, "%s: %s - [ERROR] Not enough memory !!!\n", __FILE__, __FUNCTION__); 
 return(s_Return);
}

static t_MZ_Buffer *_MZ_PushBuffer(t_MZ_Buffer *s_HANDLE_Buffer, void *s_Data, unsigned long s_Length)
{
 t_MZ_BufferNode *s_AllocNode;
 s_AllocNode = (t_MZ_BufferNode *)MZ_Alloc(sizeof(t_MZ_BufferNode)); 
 if(s_AllocNode)
 {
  s_AllocNode->Next    = (t_MZ_BufferNode *)0;
  s_AllocNode->Current = 0lu;
  s_AllocNode->Size    = s_Length; 
  s_AllocNode->Data    = s_Data;
  if(s_HANDLE_Buffer->Head)
  {
   t_MZ_BufferNode *s_TraceBufferNode = s_HANDLE_Buffer->Head;
   while(s_TraceBufferNode->Next)s_TraceBufferNode = s_TraceBufferNode->Next;
   s_TraceBufferNode->Next = s_AllocNode;
  }
  else s_HANDLE_Buffer->Head = s_AllocNode;     
 } 
 else fprintf(stderr, "%s: %s - [ERROR] Not enough memory (s_AllocNode) !!!\n", __FILE__, __FUNCTION__); 
 return(s_HANDLE_Buffer); 
}

t_MZ_Buffer *MZ_PushBuffer(t_MZ_Buffer *s_HANDLE_Buffer, void *s_Data, unsigned long s_Length)
{
 if(s_HANDLE_Buffer == (t_MZ_Buffer *)0)s_HANDLE_Buffer = MZ_CreateBuffer();
 if(s_HANDLE_Buffer)
 {
  if(s_Data)
  {
   unsigned long s_TrySize;	  
   while(s_Length > 0lu)
   {	   
    if(s_HANDLE_Buffer->Align == (void *)0)
    {
     s_HANDLE_Buffer->Align = (void *)MZ_Alloc(DEF_BUFFER_ALIGN);
     s_HANDLE_Buffer->AlignIndex = 0;
     if(s_HANDLE_Buffer->Align == (void *)0)
     {
      fprintf(stderr, "%s: %s - [ERROR] Align buffer drop (Not enough memory) !!!\n", __FILE__, __FUNCTION__);
      break;
     }
    }
    if((DEF_BUFFER_ALIGN - s_HANDLE_Buffer->AlignIndex) >= s_Length)s_TrySize = s_Length;
    else s_TrySize = DEF_BUFFER_ALIGN - s_HANDLE_Buffer->AlignIndex;
#if DEF_USE_ASM == (0)
    memcpy(((char *)s_HANDLE_Buffer->Align) + s_HANDLE_Buffer->AlignIndex, s_Data, s_TrySize);
#else
  __asm__ __volatile__(
   "\n\t"
   "movl %0, %%edi\n\t"
   "movl %1, %%esi\n\t"
   "movl %2, %%ecx\n\t"
   "addl %3, %%edi\n\t"
   "cld\n\t"
   "shrl $1, %%ecx\n\t"
   "jnc 0f\n\t"
   "movsb\n\t"
   "0:\n\t"
   "shrl $1, %%ecx\n\t"
   "jnc 0f\n\t"
   "movsw\n\t"
   "0:\n\t"
   "repz movsl\n\t"
   "\n\t"
   :
   : "m"(s_HANDLE_Buffer->Align), "m"(s_Data), "m"(s_TrySize), "m"(s_HANDLE_Buffer->AlignIndex)
   : "esi", "edi"
  );
#endif
    s_Length                    -= s_TrySize;
    s_HANDLE_Buffer->Total      += s_TrySize;
    s_HANDLE_Buffer->AlignIndex += s_TrySize; 
    s_Data = (void *)((unsigned char *)s_Data + s_TrySize);
    if(s_HANDLE_Buffer->AlignIndex >= DEF_BUFFER_ALIGN)
    {
     s_HANDLE_Buffer = _MZ_PushBuffer(s_HANDLE_Buffer, s_HANDLE_Buffer->Align, s_HANDLE_Buffer->AlignIndex);
     s_HANDLE_Buffer->Align = (void *)0;
     s_HANDLE_Buffer->AlignIndex = 0;
    }
   }
  }
  else fprintf(stderr, "%s: %s - [ERROR] s_Data is null !!!\n", __FILE__, __FUNCTION__);
 }
 else fprintf(stderr, "%s: %s - [ERROR] s_HANDLE_Buffer is null !!!\n", __FILE__, __FUNCTION__); 
 return(s_HANDLE_Buffer);
}

unsigned long MZ_PopBuffer(t_MZ_Buffer *s_HANDLE_Buffer, void *s_Data, unsigned long s_Length)
{
 unsigned long s_Return = 0lu;
 if(s_HANDLE_Buffer)
 {
  if(s_Length > 0lu)
  {
   unsigned long s_TrySize;
   if(s_HANDLE_Buffer->Align && s_HANDLE_Buffer->AlignIndex > 0lu && s_HANDLE_Buffer->Total < (s_Length + s_HANDLE_Buffer->AlignIndex))
   {
    s_HANDLE_Buffer = _MZ_PushBuffer(s_HANDLE_Buffer, s_HANDLE_Buffer->Align, s_HANDLE_Buffer->AlignIndex);
    s_HANDLE_Buffer->Align = (void *)0;
    s_HANDLE_Buffer->AlignIndex = 0;
   }
   while(s_Length > 0lu && s_HANDLE_Buffer->Head)
   {
    if(s_Length <= (s_HANDLE_Buffer->Head->Size - s_HANDLE_Buffer->Head->Current))s_TrySize = s_Length;
    else s_TrySize = (s_HANDLE_Buffer->Head->Size - s_HANDLE_Buffer->Head->Current);
    if(s_Data)
    {
     memcpy(((char *)s_Data) + s_Return, ((char *)s_HANDLE_Buffer->Head->Data) + s_HANDLE_Buffer->Head->Current, s_TrySize);
    }
    s_Return += s_TrySize;
    s_Length -= s_TrySize;
    s_HANDLE_Buffer->Head->Current += s_TrySize;    
    if(s_HANDLE_Buffer->Head->Current >= s_HANDLE_Buffer->Head->Size)
    {
     t_MZ_BufferNode *s_TraceBufferNode = s_HANDLE_Buffer->Head;
     s_HANDLE_Buffer->Head = s_HANDLE_Buffer->Head->Next; 
     s_TraceBufferNode->Next = (t_MZ_BufferNode *)0;
     MZ_FlushBuffer(s_TraceBufferNode);
    }	
   }
   s_HANDLE_Buffer->Total -= s_Return;  
  }     
 }
 return(s_Return); 
}

static t_MZ_BufferNode *MZ_FlushBuffer(t_MZ_BufferNode *s_BufferNode)
{
 t_MZ_BufferNode *s_TraceBufferNode;   
 while(s_BufferNode)
 {
  s_TraceBufferNode = s_BufferNode;
  s_BufferNode = s_BufferNode->Next;  
  if(s_TraceBufferNode->Data)MZ_Free(s_TraceBufferNode->Data);
  MZ_Free(s_TraceBufferNode); 
 }
 return(s_BufferNode); 
}

t_MZ_Buffer *MZ_DestroyBuffer(t_MZ_Buffer *s_HANDLE_Buffer)
{
 if(s_HANDLE_Buffer)
 {
  MZ_FlushBuffer(s_HANDLE_Buffer->Head);
  if(s_HANDLE_Buffer->Align)
  {
   MZ_Free(s_HANDLE_Buffer->Align);
   s_HANDLE_Buffer->Align = (void *)0;
   s_HANDLE_Buffer->AlignIndex = 0;
  }
  MZ_Free(s_HANDLE_Buffer);
  s_HANDLE_Buffer = (t_MZ_Buffer *)0; 
 }
 return(s_HANDLE_Buffer); 
}

unsigned long MZ_CheckBuffer(t_MZ_Buffer *s_HANDLE_Buffer)
{
 if(s_HANDLE_Buffer)return(s_HANDLE_Buffer->Total);
 return(0lu); 
}

static void *MZ_AllocString(void *s_Data, int s_Length)
{
 void *s_Return = (void *)0;
 if(s_Data)
 {
  if(s_Length <= 0)s_Length = strlen(s_Data);
  else
  {
   int s_ScanLength = 0;
   while(s_ScanLength < s_Length)
   {
    if(*(((char *)s_Data) + s_ScanLength) == '\0')
    {
     s_Length = s_ScanLength + 1;	    
     break;
    }
    s_ScanLength++; 
   }
  }   
  s_Return = (void *)MZ_Alloc(s_Length + 1);
  if(s_Return)
  {
   memcpy(s_Return, s_Data, s_Length);
   *(((char *)s_Return) + s_Length) = '\0';
  }
 }
 return(s_Return);
}

static int MZ_CompareString(void *s_Left, void *s_Right, int s_Length)
{
 int s_Return = 0;
 if(s_Left && s_Right)
 {
  if(s_Length <= 0)
  {
   int s_LeftLength, s_RightLength;
   s_LeftLength = strlen(s_Left);
   s_RightLength = strlen(s_Right);
   s_Return = s_LeftLength - s_RightLength;
   if(s_Return != 0)return(s_Return);
   s_Length = strlen(s_Left);
  }	  
  while(s_Length--)
  {
   if(*((char *)s_Left) == '\0' && *((char *)s_Right) == '\0')break;  
   else if(*((char *)s_Left) == '\0')
   {
    s_Return = (-1);	   
    break;  
   }
   else if(*((char *)s_Right) == '\0')
   {
    s_Return = 1;	   
    break;  
   }
   if(*((char *)s_Left) != *((char *)s_Right))
   {
    s_Return = (-2);	   
    break;	 
   }
   s_Left = (void *)((unsigned char *)s_Left + 1);
   s_Right = (void *)((unsigned char *)s_Right + 1);
  }	 
 }
 else s_Return = (-1); 
 return(s_Return); 
}

static void *MZ_SkipString(void *s_String, void *s_It, e_MZ_Parser s_Flag)
{
 if(s_String && s_It)
 {
  int s_LengthIt;
  s_LengthIt = strlen(s_It);  
  if(s_Flag == E_PARSER_RIGHT)
  {
   int s_IndexString;
   s_IndexString = strlen(s_String);
   while(s_IndexString >= s_LengthIt)
   {
    s_IndexString -= s_LengthIt;
    if(MZ_CompareString(((char *)s_String) + s_IndexString, s_It, (-1)) == 0)*(((char *)s_String) + s_IndexString) = '\0';
    else break;
   }
  }
  else if(s_Flag == E_PARSER_LEFT || s_Flag == E_PARSER_CENTER)
  {
   int s_IndexString = 0, s_IndexIt = 0, s_OK = 0;
   while(*(((char *)s_String) + s_IndexString))
   {
    if(MZ_CompareString(((char *)s_String) + s_IndexString, s_It, s_LengthIt) == 0 && s_OK == 0)s_IndexString += s_LengthIt;
    else
    {
     if(s_Flag == E_PARSER_LEFT)
     {
      if(s_IndexString == s_IndexIt)break;
      s_OK = 1;
     }
     if(s_IndexString != s_IndexIt)*(((char *)s_String) + s_IndexIt) = *(((char *)s_String) + s_IndexString);
     s_IndexString++;
     s_IndexIt++;
    } 
   }
   if(s_IndexString != s_IndexIt)*(((char *)s_String) + s_IndexIt) = '\0';
  }
  /* else fprintf(stderr, "%s: %s - [ERROR] Unknown s_Flag !!!\n", __FILE__, __FUNCTION__); */
 }	 
 return(s_String);
}

#endif 

/* End of source */
