/*
 Copyright (C) Information Equipment co.,LTD.
 All rights reserved.
 Code by JaeHyuk Cho <mailto:minzkn@infoeq.com>
 CVSTAG="$Header: /var/cvsroot/main/pmp3_tiny/main.c,v 1.1.1.1 2006/01/12 06:14:27 minzkn Exp $"
*/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/soundcard.h>
#include <pthread.h>
#include "mp3_ext.h"



#if defined(HZ) /* #include <sys/param.h> */
# define def_pmp3_hz                   (HZ)
#else
# define def_pmp3_hz                   (100)
#endif

#define def_pmp3_push_block_size       ( 32 << 10) /*  32 KBytes */
#define def_pmp3_pop_block_size        ( 32 << 10) /*  32 KBytes */
#define def_pmp3_info_block_size       ( 32 << 10) /*  32 KBytes */
#define def_pmp3_limit_buffer          (512 << 10) /* 512 KBytes */

#define def_pmp3_master_volume         (100)
#define def_pmp3_pcm_volume            ( 76)

#define def_pmp3_dsp_device            "/dev/dsp"
#define def_pmp3_mixer_device          "/dev/mixer"

volatile int g_pmp3_decode_break = 0;

struct ts_pmp3_dsp_control
{
 volatile int dsp_break;
};

static void pmp3_signal(int s_signal)
{
 switch(s_signal)
 {
  case SIGINT:
  case SIGTERM:
  case SIGQUIT:
       g_pmp3_decode_break = 1;
       (void)signal(s_signal, pmp3_signal);
       break;
  case SIGSEGV:
  case SIGFPE:
  case SIGILL:
       g_pmp3_decode_break = 1;
       (void)signal(s_signal, SIG_DFL);
       (void)raise(s_signal);
       exit(1);
       break;
  default: break;
 }
}

static void pmp3_load_balance(void)
{
 struct timeval s_timeval = {0l, 1000000l / ((long)def_pmp3_hz)};
 (void)select(0, (fd_set *)0, (fd_set *)0, (fd_set *)0, (struct timeval *)(&s_timeval));
}

static void * pmp3_dsp(void *s_argument)
{
 struct ts_pmp3_dsp_control *s_dsp_control = (struct ts_pmp3_dsp_control *)s_argument;
 int s_handle, s_pop_size, s_frequency, s_prev_frequency = 0, s_format = AFMT_S16_NE, s_channels = 2;
 unsigned char s_buffer[ def_pmp3_pop_block_size ];
 s_handle = open(def_pmp3_dsp_device, O_RDWR /* | O_SYNC */);
 if(s_handle != (-1))
 {
  if(ioctl(s_handle, SNDCTL_DSP_SETFMT, &s_format) == (-1))(void)fprintf(stderr, "can not supported 16bits per sample !\n");
  if(ioctl(s_handle, SNDCTL_DSP_CHANNELS, &s_channels) == (-1))(void)fprintf(stderr, "can not supported stereo !\n");
 }
 else (void)perror(def_pmp3_dsp_device);
/*#if 0 startup buffer - OPTIONAL */
#if DEBUG_ARM
 do
 {
  if(MZ_MP3_DecoderCheckBuffer() >= sizeof(s_buffer))break;
  pmp3_load_balance();
 }while(s_dsp_control->dsp_break == 0);
#endif
 do
 {
  s_pop_size = MZ_MP3_DecoderPop((void *)(&s_buffer[0]), (int)sizeof(s_buffer));
  if((s_pop_size > 0) && (s_handle != (-1)))
  {
   s_frequency = MZ_MP3_DecoderFrequency();
   if(s_frequency > 0)
   {
    if(s_frequency != s_prev_frequency)
    {
     (void)fprintf(stderr, "frequency detected : %d Hz\n", s_frequency);
     if(ioctl(s_handle, SNDCTL_DSP_SPEED, &s_frequency) == (-1))(void)fprintf(stderr, "can not change frequency ! (frequency=%d Hz)\n", s_frequency);
     s_prev_frequency = s_frequency;
    }
    if(write(s_handle, (void *)(&s_buffer[0]), (size_t)s_pop_size) == ((ssize_t)(-1)))(void)perror("dsp write");
    else (void)fprintf(stderr, "play %d bytes (BUFFER=%d%%)\n", s_pop_size, MZ_MP3_DecoderCheckBuffer() * 100 / def_pmp3_limit_buffer);
   }
  }
  else pmp3_load_balance();
 }while(s_dsp_control->dsp_break == 0);
 if(s_handle != (-1))(void)close(s_handle);
 return(s_argument);
}

static void pmp3_dump_info(int s_handle)
{
 t_MP3 *s_mp3;
 unsigned char s_buffer[ def_pmp3_info_block_size ];
 ssize_t  s_read_bytes;
 s_mp3 = MZ_CreateMP3();
 if(s_mp3 != ((t_MP3 *)0))
 {
  s_read_bytes = read(s_handle, (void *)(&s_buffer[0]), (size_t)sizeof(s_buffer));
  if(s_read_bytes > 0)s_mp3 = MZ_PushMP3(s_mp3, (void *)(&s_buffer[0]), (int)s_read_bytes);
  if(s_mp3->IsID3 == 0)
  {
   if(lseek(s_handle, -((off_t)sizeof(s_buffer)), SEEK_END) != ((off_t)(-1)))
   {
    s_read_bytes = read(s_handle, (void *)(&s_buffer[0]), (size_t)sizeof(s_buffer));
    if(s_read_bytes > 0)s_mp3 = MZ_PushMP3(s_mp3, (void *)(&s_buffer[0]), (int)s_read_bytes);
   }
  }
  if(s_mp3->IsID3 == 1)
  {
   if(s_mp3->Title  )(void)fprintf(stderr, " Title    : \"%s\"\n", s_mp3->Title);
   if(s_mp3->Artist )(void)fprintf(stderr, " Artist   : \"%s\"\n", s_mp3->Artist);
   if(s_mp3->Album  )(void)fprintf(stderr, " Album    : \"%s\"\n", s_mp3->Album);
   if(s_mp3->Year   )(void)fprintf(stderr, " Year     : \"%s\"\n", s_mp3->Year);
   if(s_mp3->Comment)(void)fprintf(stderr, " Comment  : \"%s\"\n", s_mp3->Comment);
   if(s_mp3->Genre  )(void)fprintf(stderr, " Genre    : \"%s\"\n", s_mp3->Genre);
  }
  else (void)fprintf(stderr, " not found ID3\n");
  s_mp3 = MZ_DestroyMP3(s_mp3);
  (void)lseek(s_handle, (off_t)0, SEEK_SET);
 }
}

static int pmp3_decode(const char *s_filename)
{
 int s_result = (-1), s_handle;
 unsigned char s_buffer[ def_pmp3_push_block_size ];
 ssize_t  s_read_bytes;
 pthread_t s_pthread;
 struct ts_pmp3_dsp_control s_dsp_control;
 (void)fprintf(stderr, "pmp3_decode: filename=\"%s\"\n", s_filename);
 s_handle = open(s_filename, O_RDONLY /* | O_LARGEFILE */);
 if(s_handle != (-1))
 {
  (void)pmp3_dump_info(s_handle); /* OPTIONAL */
  MZ_MP3_DecoderOpen();
  s_dsp_control.dsp_break = 0;
  if(pthread_create((pthread_t *)(&s_pthread), (pthread_attr_t *)0, pmp3_dsp, (void *)(&s_dsp_control)) == 0)
  {
   s_result = 0;
   do
   {
    if(MZ_MP3_DecoderCheckBuffer() < def_pmp3_limit_buffer)
    {
     s_read_bytes = read(s_handle, (void *)(&s_buffer[0]), (size_t)sizeof(s_buffer));
     if(s_read_bytes > ((ssize_t)0))s_result += MZ_MP3_DecoderPush((void *)(&s_buffer[0]), (int)s_read_bytes); 
     else
     {
      while(MZ_MP3_DecoderCheckBuffer() > 0)pmp3_load_balance(); /* Wait for DSP play */
      break;
     }
    }
    else pmp3_load_balance();
   }while(g_pmp3_decode_break == 0);
   s_dsp_control.dsp_break = 1;
   (void)pthread_join(s_pthread, (void **)0);
  }
  else (void)fprintf(stderr, "can not create thread !\n");
  MZ_MP3_DecoderClose();
  (void)close(s_handle);
 }
 else (void)perror(s_filename);
 return(s_result);
}

static int pmp3_set_mixer(int s_channel, int s_left, int s_right)
{
 int s_result = (-1), s_handle, s_level;
 s_handle = open(def_pmp3_mixer_device, O_RDONLY);
 if(s_handle != (-1))
 {
  s_level = (s_left << 8) | s_right;
  if(ioctl(s_handle, MIXER_WRITE(s_channel), &s_level, sizeof(int)) == 0)s_result = 1;
  else (void)perror("mixer");
  (void)close(s_handle);
 }
 else (void)perror("mixer");
 return(s_result);
}	

int main(int s_argc, char **s_argv)
{
 int s_result = 0, s_index;
 (void)signal(SIGINT , pmp3_signal);
 (void)signal(SIGTERM, pmp3_signal);
 (void)signal(SIGQUIT, pmp3_signal);
 (void)signal(SIGSEGV, pmp3_signal);
 (void)signal(SIGFPE , pmp3_signal);
 (void)signal(SIGILL , pmp3_signal);
 if(s_argc >= 2)
 {
  (void)pmp3_set_mixer(SOUND_MIXER_VOLUME, def_pmp3_master_volume, def_pmp3_master_volume); /* Master volume */
  (void)pmp3_set_mixer(SOUND_MIXER_PCM   , def_pmp3_pcm_volume   , def_pmp3_pcm_volume   ); /* 67 ~ 76 : Maybe 0dB */
  for(s_index = 1;(s_index < s_argc) && (g_pmp3_decode_break == 0);s_index++)
  {
   if(pmp3_decode(s_argv[s_index]) == (-1)){ s_result = 1; break; }
  }
 }
 else (void)fprintf(stderr, "Usage: %s <mp3 file> [...]\n", s_argv[0]);
 return(s_result);
}

/* End of source */
