/*
  Copyright (C) JAEHYUK CHO
  All rights reserved.
  Code by JaeHyuk Cho <mailto:minzkn@minzkn.com>
*/

#include <sys/types.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <math.h>
#include <signal.h>

#include <linux/fb.h>

#define DEF_2PI (2.0 * 3.14159265358)

static int                        g_FB_Handle, g_FB_Cx, g_FB_Cy, g_FB_ScreenSize;
static struct fb_fix_screeninfo   g_FB_FIX;
static struct fb_var_screeninfo   g_FB_VAR;
static void                      *g_FB_Map;
static void (*DrawPixel)(int, int, int);

volatile int g_break_clock = 0;

static void clock_signal(int s_signal)
{
 g_break_clock = 1;
 (void)signal(s_signal, clock_signal);
}

static void (clock_load_balance)(void)
{
#if 1
 struct timeval s_timeval = {0l, 1000000l / 100l /* hz */};
 (void)select(0, (fd_set *)0, (fd_set *)0, (fd_set *)0, (struct timeval *)(&s_timeval));
#endif
}

static void DrawPixel8(int s_Color, int s_x, int s_y)
{
 *(((unsigned char *)g_FB_Map) + (s_y * g_FB_FIX.line_length) + s_x) = (unsigned char)s_Color;
}

static void DrawPixel16(int s_Color, int s_x, int s_y)
{
 *((unsigned short *)(((unsigned char *)g_FB_Map) + (s_y * g_FB_FIX.line_length) + (s_x << 1))) = (unsigned short)s_Color;
}

static void DrawPixel24(int s_Color, int s_x, int s_y)
{
 unsigned char *s_FB_Ptr = ((unsigned char *)g_FB_Map) + (s_y * g_FB_FIX.line_length) + ((s_x << 1) + s_x);
 *((unsigned short *)s_FB_Ptr) = (unsigned short)s_Color;
 *(s_FB_Ptr + sizeof(unsigned short)) = (unsigned char)(s_Color >> 16); 
}

static void DrawPixel32(int s_Color, int s_x, int s_y)
{
 *((unsigned long *)(((unsigned char *)g_FB_Map) + (s_y * g_FB_FIX.line_length) + (s_x << 2))) = (unsigned long)s_Color;
}

static void PinXY(int s_Angle, int *s_x, int *s_y, int s_r)
{ /* s_To(Grid) : X = s_From(Grid) : s_Current */
 double s_Value = ((DEF_2PI * (double)(s_Angle % 360)) / 360.0);
 *(s_x) = (int)(sin(s_Value) * s_r); *(s_y) = (int)(-cos(s_Value) * s_r); 
}

static void __DrawLine__(int s_Color, float s_x1, float s_y1, float s_x2, float s_y2, int s_Level)
{ /* Call by call level 16 optimize draw line : JaeHyuk algorithm ^^ */
 float s_cx = (s_x1 + s_x2) / 2.0, s_cy = (s_y1 + s_y2) / 2.0;
 if(((int)s_x1 == (int)s_x2 && (int)s_y1 == (int)s_y2) || s_Level > 16)DrawPixel(s_Color, (int)s_cx, (int)s_cy);
 else
 {
  s_Level++;	
  __DrawLine__(s_Color, s_x1, s_y1, s_cx, s_cy, s_Level); __DrawLine__(s_Color, s_cx, s_cy, s_x2, s_y2, s_Level);
 }
}

static void DrawLine(int s_Color, int s_x1, int s_y1, int s_x2, int s_y2)
{
 __DrawLine__(s_Color, (float)s_x1, (float)s_y1, (float)s_x2, (float)s_y2, 0);
}

static void DrawCircle(int s_Color, int s_x, int s_y, int s_r)
{
 double s_Pi, s_Grid = 1.0 / ((double)s_r), s_sinX, s_cosY; 
 for(s_Pi = 0.0;s_Pi < (DEF_2PI / 4.0);s_Pi += s_Grid)
 {
  s_sinX = sin(s_Pi) * (double)s_r; s_cosY = -cos(s_Pi) * (double)s_r;	 
  DrawPixel(s_Color, (int)(s_x + s_sinX), (int)(s_y + s_cosY));
  DrawPixel(s_Color, (int)(s_x - s_sinX), (int)(s_y + s_cosY));
  DrawPixel(s_Color, (int)(s_x + s_sinX), (int)(s_y - s_cosY));
  DrawPixel(s_Color, (int)(s_x - s_sinX), (int)(s_y - s_cosY));
 }
}

static void Pin(int s_Color, int s_Angle, int s_Type)
{
 int s_x1, s_y1, s_x2, s_y2, s_x3, s_y3;
 int s_Size;
 s_Size = g_FB_ScreenSize >> 1;
 if(s_Type == 0)       s_Size -= ((g_FB_ScreenSize >> 1) >> 3);
 else if(s_Type == 10) s_Size -= ((g_FB_ScreenSize >> 1) >> 3);
 else if(s_Type == 20) s_Size -= ((g_FB_ScreenSize >> 1) >> 2);
 PinXY(s_Angle, &s_x1, &s_y1, s_Size >> 2); PinXY(s_Angle, &s_x2, &s_y2, s_Size);
 DrawLine(s_Color, g_FB_Cx - s_x1, g_FB_Cy - s_y1, g_FB_Cx + s_x2, g_FB_Cy + s_y2);
 if(s_Type > 0)
 {
  PinXY(s_Angle +  90, &s_x3, &s_y3, s_Type);
  DrawLine(s_Color, g_FB_Cx - s_x1, g_FB_Cy - s_y1, g_FB_Cx + s_x3, g_FB_Cy + s_y3);
  DrawLine(s_Color, g_FB_Cx + s_x2, g_FB_Cy + s_y2, g_FB_Cx + s_x3, g_FB_Cy + s_y3);
  DrawLine(s_Color, g_FB_Cx - s_x1, g_FB_Cy - s_y1, g_FB_Cx - s_x3, g_FB_Cy - s_y3);
  DrawLine(s_Color, g_FB_Cx + s_x2, g_FB_Cy + s_y2, g_FB_Cx - s_x3, g_FB_Cy - s_y3);
 }
}

int main(int s_argc, char **s_argv)
{
 (void)signal(SIGINT, clock_signal);
 printf("fbclock v1.0.1 - Code by JaeHyuk Cho <minzkn@minzkn.com>\n\n");
 g_FB_Handle = open(s_argc >= 2 ? s_argv[1] : "/dev/fb0", O_RDWR);
 if(g_FB_Handle >= 0)
 {
  if(ioctl(g_FB_Handle, FBIOGET_FSCREENINFO, &g_FB_FIX) == 0)
  {
   if(ioctl(g_FB_Handle, FBIOGET_VSCREENINFO, &g_FB_VAR) == 0)
   {
    size_t s_page_size;
    size_t s_alignment_miss_size;
    s_page_size = (size_t)(4 << 10);
    s_alignment_miss_size = ((size_t)g_FB_FIX.smem_start) % s_page_size;
    g_FB_Map = mmap((void *)0,
     g_FB_FIX.line_length * g_FB_VAR.yres + ((s_alignment_miss_size != ((size_t)0)) ? s_page_size : 0), PROT_READ | PROT_WRITE, MAP_SHARED, g_FB_Handle, 0);    
    if(g_FB_Map != (void *)(-1))
    { 
     if(s_alignment_miss_size != ((size_t)0))
     {
      g_FB_Map = ((unsigned char *)g_FB_Map) + s_alignment_miss_size;
      (void)fprintf(stdout, "fix alignment map\n");
     }
     switch(g_FB_VAR.bits_per_pixel)
     {
      case  8: DrawPixel = DrawPixel8;  break;
      case 16: DrawPixel = DrawPixel16; break;
      case 24: DrawPixel = DrawPixel24; break;
      case 32: DrawPixel = DrawPixel32; break;
      default: DrawPixel = (void *)0;   break;
     }
     g_FB_Cx = g_FB_VAR.xres >> 1; g_FB_Cy = g_FB_VAR.yres >> 1;
     g_FB_ScreenSize = g_FB_VAR.xres < g_FB_VAR.yres ? g_FB_VAR.xres : g_FB_VAR.yres;
     if(DrawPixel)
     {
      int s_PreSecond = 0, s_PreMinute = 0, s_PreHour = 0, s_Count, s_Index, s_x, s_y;
      time_t s_UTC;
      struct tm *s_LocalTime;
      s_Count = 0;
      do
      {
       s_UTC = time((time_t *)0);
       s_LocalTime = localtime(&s_UTC);
       if(s_LocalTime->tm_sec != s_PreSecond)
       {
	if(s_LocalTime->tm_min != s_PreMinute)
	{
         if(s_LocalTime->tm_hour != s_PreHour)
	 {
	  Pin(0, s_PreHour * 30, 20);	       
	  s_PreHour = s_LocalTime->tm_hour; 
	 }
	 Pin(0, s_PreMinute * 6, 10);	       
	 s_PreMinute = s_LocalTime->tm_min;
	}
        Pin(0, s_PreSecond * 6, 0);	       
        Pin(0xf00f, s_LocalTime->tm_hour * 30, 20);	       
        Pin(0x0ff0, s_LocalTime->tm_min  *  6, 10);	       
        Pin(0xffff, s_LocalTime->tm_sec  *  6, 0);	      
        DrawCircle(0x0fff, g_FB_Cx, g_FB_Cy, 5);	
        s_PreSecond = s_LocalTime->tm_sec;
       }
       do
       {
	int s_tick;
	for(s_tick = 0;s_tick < 10;s_tick++)clock_load_balance();
       }while(0);
       PinXY(s_Count, &s_x, &s_y, (g_FB_ScreenSize >> 1) - (g_FB_ScreenSize >> 5));
       s_Index = (s_Count % 30) ? 5 : 10;
       DrawCircle(0xffff, g_FB_Cx + s_x, g_FB_Cy + s_y, s_Index);	
       DrawCircle(0xffff, g_FB_Cx - s_x, g_FB_Cy + s_y, s_Index);	
       DrawCircle(0xffff, g_FB_Cx + s_x, g_FB_Cy - s_y, s_Index);	
       DrawCircle(0xffff, g_FB_Cx - s_x, g_FB_Cy - s_y, s_Index);
       s_Count+=6;
       if(s_Count > 90)s_Count = 0;
      }while(g_break_clock == 0);
     }
     else fprintf(stderr, "Err: %d\n", __LINE__);
     munmap(g_FB_Map, g_FB_FIX.line_length * g_FB_VAR.yres);
    }
    else fprintf(stderr, "Err: %d\n", __LINE__);
   }
   else fprintf(stderr, "Err: %d\n", __LINE__);
  }
  else fprintf(stderr, "Err: %d\n", __LINE__);
  close(g_FB_Handle);	 
 }
 else fprintf(stderr, "Err: %d\n", __LINE__);
 (void)fprintf(stdout, "End of clock\n");
 return(0);
}

/* vim: set expandtab: */
/* End of source */ 
