/*
 Copyright (C) JAEHYUK CHO
 All rights reserved.
 Code by JaeHyuk Cho <mailto:minzkn@minzkn.com>
*/

#if !defined(__def_mzapi_source_mzudev_c__)
#define __def_mzapi_source_mzudev_c__ "mzudev.c"

#if !defined(_GNU_SOURCE)
# define _GNU_SOURCE
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <mntent.h>

#include <linux/netlink.h>

#include "mzudev.h"

#define def_mzudev_message_buffer_size (4 << 10)

static void *mzudev_alloc(size_t s_size);
static void *mzudev_free(void *s_ptr);
static char * mzudev_get_word_sep(int s_is_skip_space, const char *s_sep, char **s_sep_string);
static int mzudev_netlink_uevent_touch_recursive(int s_depth, const char *s_pathname, const char *s_name, struct stat *s_stat);
static int mzudev_netlink_uevent_touch(const char *s_pathname);

int mzudev_open_netlink(void);
int mzudev_close_netlink(int s_socket);
int mzudev_check_event(int s_socket, int s_msec);
mzudev_event_t *mzudev_get_event(int s_socket, int s_msec);
mzudev_event_t *mzudev_update_event(mzudev_event_t *s_head, mzudev_event_t *s_event);
mzudev_event_t *mzudev_free_event(mzudev_event_t *s_event);
char *mzudev_get_env(mzudev_event_t *s_event, const char *s_name);

static void *mzudev_alloc(size_t s_size)
{
    return(malloc(s_size));
}

static void *mzudev_free(void *s_ptr)
{
    free(s_ptr);

    return((void *)0);
}

static char * mzudev_get_word_sep(int s_is_skip_space, const char *s_sep, char **s_sep_string)
{
    unsigned char *s_string, *s_result, *s_sep_ptr;

    if(s_is_skip_space == 0) {
        s_result = s_string = (unsigned char *)(*(s_sep_string));
        while(*(s_string) != 0u) {
            s_sep_ptr = (unsigned char *)s_sep;
            while((*(s_string) != *(s_sep_ptr)) && (*(s_sep_ptr) != 0u)) {
                s_sep_ptr++;
            }
            if(*(s_string) == *(s_sep_ptr)) {
                break;
            }
            ++s_string;
        }
    }
    else {
        s_string = (unsigned char *)(*(s_sep_string));
        while(isspace(*s_string) != 0) {
            s_string++;
        }
        s_result = s_string;
        while(*(s_string) != 0u) {
            s_sep_ptr = (unsigned char *)s_sep;
            while((*(s_string) != *(s_sep_ptr)) && (*(s_sep_ptr) != 0u)) {
                s_sep_ptr++;
            }
            if(*(s_string) == *(s_sep_ptr)) {
                break;
            }
            if(isspace(*s_string) != 0) {
                s_string++;
            }
            else {
                ++s_string;
            }
        }
    }
    
    *(s_sep_string) = (char *)s_string;

    return((char *)s_result);
}

static int mzudev_netlink_uevent_touch_recursive(int s_depth, const char *s_pathname, const char *s_name, struct stat *s_stat)
{
    int s_check;

    DIR *s_dir;
    struct dirent *s_dirent;

    char *s_current_pathname;
	
    if(S_ISDIR(s_stat->st_mode) == 0) {
        if(strcmp(s_name, "uevent") == 0) {
            FILE *s_touch_fp;
            
            s_touch_fp = fopen(s_pathname, "w");
            if(s_touch_fp == ((FILE *)0)) {
                return(0);
            }

            (void)fprintf(s_touch_fp, "1\n");

            fclose(s_touch_fp);
        }
 
        return(0);
    }
            
    s_dir = opendir(s_pathname);
    if(s_dir == ((DIR *)0)) {
        return(-1);
    }

    for(;;) {
        s_dirent = readdir(s_dir);
        if(s_dirent == ((struct dirent *)0)) {
            break;
        }

        s_current_pathname = mzudev_alloc(strlen(s_pathname) + 1 + strlen(s_dirent->d_name) + 1);
        if(s_current_pathname == ((char *)0)) {
            continue;
        }
        (void)sprintf(s_current_pathname, "%s/%s", s_pathname, s_dirent->d_name);
 
        s_check = (s_depth <= 0) ? stat(s_current_pathname, s_stat) : lstat(s_current_pathname, s_stat);
        if(s_check == (-1)) {
            mzudev_free((void *)s_current_pathname);
            continue;
        }

        if((S_ISDIR(s_stat->st_mode) != 0) && ((strcmp(s_dirent->d_name, ".") == 0) || (strcmp(s_dirent->d_name, "..") == 0))) {
            mzudev_free((void *)s_current_pathname);
            continue;
        }

        (void)mzudev_netlink_uevent_touch_recursive(s_depth + 1, s_current_pathname, s_dirent->d_name, s_stat);

        mzudev_free((void *)s_current_pathname);
    }

    (void)closedir(s_dir);

    return(0);
}

static int mzudev_netlink_uevent_touch(const char *s_pathname)
{
    struct stat s_stat;
    size_t s_offset;

    if(s_pathname == ((const char *)0)) {
        return(-1);
    }
 
    if(stat(s_pathname, (struct stat *)(&s_stat)) == (-1)) {
        return(-1);
    }
        
    s_offset = strlen(s_pathname);
    while(s_offset > 0) {
        --s_offset;
        if(s_pathname[s_offset] != '/') {
            break;
        }
    }
    do {
        if(s_pathname[s_offset] == '/') {
            ++s_offset;
            break;
        }
    }while((s_offset--) > 0);

    return(mzudev_netlink_uevent_touch_recursive(0, s_pathname, (const char *)(&s_pathname[s_offset]), (struct stat *)(&s_stat))); 
}

int mzudev_open_netlink(void)
{
    const int s_always_on = 1;
    int s_socket, s_socket_flags, s_check;
    struct sockaddr_nl s_sockaddr_nl;

    /* open netlink socket */
    s_socket = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_KOBJECT_UEVENT /* 15 */);
    if(s_socket == (-1)) {
        return(-1);
    }
        
    /* set close exec */
    s_socket_flags = fcntl(s_socket, F_GETFD);
    (void)fcntl(s_socket, F_SETFD, ((s_socket_flags == (-1)) ? 0 : s_socket_flags) | FD_CLOEXEC);
        
    /* bind */
    (void)memset((void *)(&s_sockaddr_nl), 0, sizeof(s_sockaddr_nl));
    s_sockaddr_nl.nl_family = AF_NETLINK;
    s_sockaddr_nl.nl_groups = 1;
    s_check = bind(s_socket, (struct sockaddr *)(&s_sockaddr_nl), (socklen_t)sizeof(s_sockaddr_nl));
    if(s_check == (-1)) {
        return(mzudev_close_netlink(s_socket));
    }
    s_check = setsockopt(s_socket, SOL_SOCKET, SO_PASSCRED, &s_always_on, (socklen_t)sizeof(s_always_on));
    if(s_check == (-1)) {
        return(mzudev_close_netlink(s_socket));
    }

#if 1L
    do {
        FILE *s_mnt;

	s_mnt = setmntent("/proc/mounts", "r");
	if(s_mnt != ((FILE *)0)) {
	    struct mntent *s_mntent;

	    for(;;) {
                s_mntent = getmntent(s_mnt);
		if(s_mntent == ((struct mntent *)0)) {
                    break;
		}
                if((s_mntent->mnt_type == ((char *)0)) || (s_mntent->mnt_dir == ((char *)0))) {
		    continue;
		}
		if(strcmp(s_mntent->mnt_type, "sysfs") == 0) {
	            char s_touch_path[ PATH_MAX ];

                    /* request re-event for block device (optional) - need root permission */
                    (void)sprintf((char *)(&s_touch_path[0]), "%s/%s", s_mntent->mnt_dir, "block");
                    (void)mzudev_netlink_uevent_touch((const char *)(&s_touch_path[0]));
                    break;         
		}
	    }

            (void)endmntent(s_mnt);
	}
    }while(0);
#endif

    return(s_socket);
}

int mzudev_close_netlink(int s_socket)
{
    /* close socket */ 
    (void)close(s_socket);

    return(-1);
}

int mzudev_check_event(int s_socket, int s_msec)
{
    int s_check;
    fd_set s_fd_rx;
    struct timeval s_timeval_local, *s_timeval;

    FD_ZERO(&s_fd_rx);
    FD_SET(s_socket, &s_fd_rx);

    if(s_msec == (-1)) {
        s_timeval = (struct timeval *)0;
    }
    else {
        s_timeval = (struct timeval *)(&s_timeval_local);
        s_timeval->tv_sec = s_msec / 1000;
        s_timeval->tv_usec = (s_msec % 1000) * 1000;
    }

    s_check = select(s_socket + 1, (fd_set *)(&s_fd_rx), (fd_set *)0, (fd_set *)0, s_timeval);
    
    return((s_check > 0) ? FD_ISSET(s_socket, &s_fd_rx) : s_check);
}

mzudev_event_t *mzudev_get_event(int s_socket, int s_msec)
{
    mzudev_event_t *s_result;

    int s_check;

    struct sockaddr_nl s_sockaddr_nl;
    unsigned char s_buffer[ def_mzudev_message_buffer_size + 1 ];
    struct iovec s_iovec;
    struct msghdr s_msghdr;
    unsigned char s_cred_msg[CMSG_SPACE(sizeof(struct ucred))];
    ssize_t s_recv_bytes;
        
    struct cmsghdr *s_cmsghdr;
    struct ucred *s_ucred;
    size_t s_offset;
    int s_env_count;

    char *s_line;

    if(s_msec >= 0) {
        s_check = mzudev_check_event(s_socket, s_msec);
        if((s_check == (-1)) || (s_check == 0)) {
            return((mzudev_event_t *)0);
        }
    }
            
    (void)memset((void *)(&s_sockaddr_nl), 0, sizeof(s_sockaddr_nl));
    s_sockaddr_nl.nl_family = AF_NETLINK;
    s_sockaddr_nl.nl_groups = 1;

    s_iovec.iov_base = &s_buffer[0];
    s_iovec.iov_len = sizeof(s_buffer) - ((size_t)1u);

    (void)memset((void *)(&s_msghdr), 0, sizeof(s_msghdr));
    s_msghdr.msg_iov = &s_iovec;
    s_msghdr.msg_iovlen = 1;
    s_msghdr.msg_control = &s_cred_msg[0];
    s_msghdr.msg_controllen = sizeof(s_cred_msg);
    s_msghdr.msg_name = &s_sockaddr_nl;
    s_msghdr.msg_namelen = sizeof(s_sockaddr_nl);

    s_recv_bytes = recvmsg(s_socket, &s_msghdr, 0);
    if(s_recv_bytes == ((ssize_t)(-1))) {
        /* revmsg error */
        return((mzudev_event_t *)0);
    }
    s_buffer[s_recv_bytes] = '\0';
    
    s_cmsghdr = CMSG_FIRSTHDR(&s_msghdr);
    s_ucred = (struct ucred *)CMSG_DATA(s_cmsghdr);
    if(s_ucred->uid != 0) {
        /* sender pid 0 is ignore */
        return((mzudev_event_t *)0);
    }
    
    s_line = (char *)(&s_buffer[0]);
    s_offset = strlen((const char *)s_line) + ((size_t)1u);
    if((s_offset < ((size_t)(/* "a@/d" */ 4 + 1))) || (s_offset >= ((size_t)s_recv_bytes))) {
        /* invalid message size */
        return((mzudev_event_t *)0);
    }

    s_env_count = (size_t)0;
    while(s_offset < ((size_t)s_recv_bytes)) {
        s_offset += strlen((const char *)(&s_buffer[s_offset])) + ((size_t)1);
        ++s_env_count;
    }

    s_result = mzudev_alloc(sizeof(mzudev_event_t) + (sizeof(mzudev_env_t) * s_env_count) + (((size_t)s_recv_bytes) + ((size_t)1)));
    if(s_result == ((mzudev_event_t *)0)) {
        /* can not allocate */
        return((mzudev_event_t *)0);
    }
    s_result->prev = (mzudev_event_t *)0;
    s_result->next = (mzudev_event_t *)0;
    s_result->env_count = s_env_count;
    s_result->env = (mzudev_env_t *)(((unsigned char *)s_result) + sizeof(mzudev_event_t));
    s_result->size = (size_t)s_recv_bytes;
    s_result->data = (unsigned char *)(&s_result->env[s_result->env_count]);

    (void)memcpy((void *)s_result->data, (const void *)(&s_buffer[0]), (size_t)s_recv_bytes);
    s_result->data[s_recv_bytes] = '\0'; 
    s_result->title = (char *)s_result->data;
    s_result->action = (char *)0;
    s_result->devpath = (char *)0;

    for(s_env_count = 0, s_offset = strlen((const char *)s_result->title) + ((size_t)1u);s_env_count < s_result->env_count;s_env_count++) {
        s_line = (char *)(&s_result->data[ s_offset ]);
        s_offset += strlen(s_line) + ((size_t)1);

        s_result->env[s_env_count].name = mzudev_get_word_sep(1, "=\n\r", (char **)(&s_line));
        if((*(s_line)) != '\0') {
            *(s_line) = '\0';
            ++s_line;
        }
        s_result->env[s_env_count].value = mzudev_get_word_sep(1, "\n\r", (char **)(&s_line));
        *(s_line) = '\0';
    }

    s_result->action = mzudev_get_env(s_result, "ACTION");
    s_result->devpath = mzudev_get_env(s_result, "DEVPATH");

    return(s_result);
}

mzudev_event_t *mzudev_update_event(mzudev_event_t *s_head, mzudev_event_t *s_event)
{
    mzudev_event_t *s_prev, *s_trace;
    char *s_devpath;

    if(s_event == ((mzudev_event_t *)0)) {
        /* invalid argument */
        return(s_head);
    }

    s_prev = (mzudev_event_t *)0;
    s_trace = s_head;
    while(s_trace != ((mzudev_event_t *)0)) {
        s_devpath = mzudev_get_env(s_trace, "DEVPATH");
	if((s_devpath != ((char *)0)) && (s_event->devpath != ((char *)0))) {
            if(strcmp(s_devpath, s_event->devpath) == 0) {
                break;
            }
	}
	s_prev = s_trace;
        s_trace = s_trace->next;
    }

    if(strcasecmp(s_event->action, "add") == 0) {
        if(s_trace == ((mzudev_event_t *)0)) {
            /* new event */
	    if(s_prev == ((mzudev_event_t *)0)) {
                s_event->prev = (mzudev_event_t *)0;
                s_event->next = (mzudev_event_t *)0;
                s_head = s_event;
            }
            else {
                s_event->prev = s_prev;
                s_event->next = (mzudev_event_t *)0;
                s_prev->next = s_event;
            }
        }
        else {
            /* replace event */
            s_event->prev = s_trace->prev;
            s_event->next = s_trace->next;
            if(s_trace->prev == ((mzudev_event_t *)0)) {
                s_head = s_event;
            }
            else {
                s_trace->prev->next = s_event;
            }
            if(s_trace->next != ((mzudev_event_t *)0)) {
                s_trace->next->prev = s_event;
            }

            s_trace->prev = (mzudev_event_t *)0;
            s_trace->next = (mzudev_event_t *)0;
            s_trace = mzudev_free_event(s_trace);
        }
    }
    else if(strcasecmp(s_event->action, "remove") == 0) {
        /* remove event */
        if(s_trace != ((mzudev_event_t *)0)) {
            s_event->prev = s_trace->prev;
            if(s_trace->prev == ((mzudev_event_t *)0)) {
                s_head = s_trace->next;
            }
            else {
                s_trace->prev->next = s_trace->next;
            }
            if(s_trace->next != ((mzudev_event_t *)0)) {
                s_trace->next->prev = s_trace->prev;
            }

            s_trace->prev = (mzudev_event_t *)0;
            s_trace->next = (mzudev_event_t *)0;
            s_trace = mzudev_free_event(s_trace);
        }

        s_event->prev = (mzudev_event_t *)0;
        s_event->next = (mzudev_event_t *)0;
        s_event = mzudev_free_event(s_event);
    }
    else {
        /* drop event */
        s_event->prev = (mzudev_event_t *)0;
        s_event->next = (mzudev_event_t *)0;
        s_event = mzudev_free_event(s_event);
    }

    return(s_head);
}

mzudev_event_t *mzudev_free_event(mzudev_event_t *s_event)
{
    mzudev_event_t *s_prev;

    while(s_event != ((mzudev_event_t *)0)) {
        s_prev = s_event;
        s_event = s_event->next;

        (void)mzudev_free((void *)s_prev);
    }

    return((mzudev_event_t *)0);
}

char *mzudev_get_env(mzudev_event_t *s_event, const char *s_name)
{
    int s_env_count;

    for(s_env_count = 0;s_env_count < s_event->env_count;s_env_count++) {
        if(s_event->env[s_env_count].name == ((char *)0)) {
            continue;
        }

        if(strcasecmp((const char *)s_event->env[s_env_count].name, s_name) == 0) {
            return(s_event->env[s_env_count].value);
        }
    }

    return((char *)0);
}

#endif

/* vim: set expandtab: */
/* End of source */
